/*
 * BinToTap
 * Convert binary file to tape file without header
 *
 * Copyright (C) Damian Walker 2012
 */

/*
 * Includes
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/*
 * Global variables
 */

char *binname = NULL; /* name of binary file */
char *tapname = NULL; /* name of tape file */
int speclength = 0; /* number of bytes to write to tape file */

/* display program usage */
int usage (char *progname) {
    printf ("Usage: %s [OPTIONS] BINARY-FILE\n", progname);
    printf ("Options:\n");
    printf ("    -t TAPE-FILE  Specify tape file name\n");
    printf ("    -l BYTES      Specify data length on tape\n");
    return 0;
}

/* initialise program from parameters */
int initialise (int argc, char **argv) {
    int count; /* for counting through parameters */
    char *dot; /* position of dot in filename */
    for (count = 1; count < argc; ++count) {
	if (strcmp (argv [count], "-t") == 0)
	    tapname = argv [++count];
	else if (strcmp (argv [count], "-l") == 0)
	    sscanf (argv [++count], "%d", &speclength);
	else
	    binname = argv [count];
    }
    if (count > argc + 1) {
	printf ("%s: error on command line\n", argv [0]);
	return 2;
    } else if (binname == NULL) {
	printf ("%s: no BIN file specified\n", argv [0]);
	return 3;
    }
    if (tapname == NULL) {
	tapname = malloc (5 + strlen (binname));
	strcpy (tapname, binname);
	if ((dot = strchr (tapname, '.')))
	    strcpy (dot, ".tap");
	else
	    strcat (tapname, ".tap");
    }
    return 0;
}

/* make the tap file */
int maketapefile (char *progname) {
    FILE *binfile; /* handle for binary input file */
    FILE *tapfile; /* handle for tap output file */
    int count = 0; /* byte count */
    int ch; /* character read from input file */
    int checksum; /* checksom for end of file */
    if (!(binfile = fopen (binname, "rb"))) {
	printf ("%s: cannot open input file %s\n", progname, binname);
	return 5;
    } else if (!(tapfile = fopen (tapname, "wb"))) {
	printf ("%s: cannot open output file %s\n", progname, tapname);
	return 6;
    }
    fputc (0, tapfile);
    fputc (0, tapfile);
    fputc (0xff, tapfile); /* flag byte */
    checksum = 0xff;
    while ((ch = fgetc (binfile)) != EOF) {
	++count;
	fputc (ch, tapfile);
	checksum = checksum ^ ch;
    }
    while (count < speclength) {
	++count;
	fputc (0, tapfile);
    }
    fputc (checksum, tapfile);
    rewind (tapfile);
    count += 2; /* take into account flag and checksum */
    fputc (count & 0xff, tapfile);
    fputc (count / 0x100, tapfile);
    fclose (tapfile);
    fclose (binfile);
    return 0;
}

/* main program */
int main (int argc, char **argv) {
    int result; /* return code */
    if (argc == 1)
	return usage (argv [0]);
    if ((result = initialise (argc, argv)))
	return result;
    return maketapefile (argv [0]);
}
