	;
	; Intergalactic Space Rescue - a puzzle game for the 16K ZX Spectrum
	; New Game and Level Ops
	;
	; This file contains the routines for starting a new game, restarting
	; the current level and moving on to the next level, each of which
	; require confirmation in case of being selected accidentally. Also
	; in this section is the code to choose a level to complete.
	;
	; Copyright (C) Damian Walker 2012
	; Created 27-Dec-2012
	;

	;
	; mission restart routine
	;

	; initialise ship positions
.startm	ld a,(rscini)	; load accumulator from rescue ship position
	ld (rescxy),a	; store as current position
	ld a,18		; restore fuel to full level
	ld (shfuel),a	;
	call dr_map	; draw the map

	; finished
	jp maingm	; back to main game

	;
	; next mission routine
	;

	; check player has finished this mission
.nextmn	ld a,(beacxy)	; where is the beacon?
	ld b,a		; store this information
	ld a,(rescxy)	; where is the rescue ship?
	cp b		; is it at the beacon?
	jr nz,incomp	; no, so this mission is incomplete

	; check player not finished last mission
	ld a,(mssion)	; lower byte of last mission ...
	cp $e7		; ... should be 231
	jr nz,admiss	; no, this is not the last mission
	ld a,(mssion+1)	; upper byte of last mission ...
	cp $03		; ... should be 3
	jr nz,admiss	; no, this is not the last mission
	ld hl,gmover	; point to "game over" message
	jr nexter	; go to print it
.incomp	ld hl,incmsg	; point to "mission incomplete" message
.nexter	call notprt	; display it as a notification
	jp maingm	; go back to main game

	; advance the mission
.admiss	ld hl,(mssion)	; load the mission
	inc hl		; advance to the next mission
	ld a,(shfuel)	; what fuel have we left?
	ld e,a		;
	ld d,0		; we need this as a 16-bit value
	add hl,de	; add it to the mission number
	ld a,h		; approximately how far are we through the levels?
	cp $03		; are we near the end?
	jr nz,svmiss	; no, continue
	ld a,l		; exactly what mission have we advanced to?
	cp $e8		; have we shot past the last mission?
	jr c,svmiss	; no, continue
	ld hl,999	; yes, so we'll content ourselves with the last mission
.svmiss	ld (mssion),hl	; save the mission number

	; all done
	jp newlvl	; back to new level routine before main game loop

	;
	; Mission Information Routine
	;

	; simply load HL with the mission number and call mission print sub
.msinfo	ld hl,(mssion)	; recall the mission number
	call gncode	; generate mission code
	call mssprt	; print the information
	ld hl,($5c78)	; get random number from timer
	ld (randsd),hl	; save as next seed
	jp maingm	; back to game

	; display the mission number/code as a notification
	; inputs:
	;   HL = mission number
.mssprt	push bc		; store soon-to-be-corrupted registers
	push de		;
	ld de,newmis+10	; point to digits of new mission message
	ld bc,100	; calculate the 100s
	call ldigit	; 
	ld bc,10	; calculate the 10s
	call ldigit	;
	ld a,l		; calculate the 1s
	add a,$30	;
	ld (de),a	;
	ld a,(lvcode)	; get level code
	ld l,a		;
	ld de,newmis+20	; point to pass-code of new mission message
	ld bc,10	; calculate the 10s
	call ldigit	;
	ld a,l		; calculate the 1s
	add a,$30	;
	ld (de),a	;
	ld hl,newmis	; point to new mission message
	call notprt	; print the notification
	pop de		; restore corrupted registers
	pop bc		;
	ret		; all done

	; calculate level digit
	; inputs:
	;   BC = digit to calculate (100, 10, 1)
	;   DE = address to save digit's ASCII value
	;   HL = number we're interested in - should not exceed 10*BC
	; outputs:
	;   A = ASCII value of digit
	;   DE = location to save next digit, if any
	;   (DE) = ASCII value of digit
.ldigit	xor a		; initialise counter
.tstdig	sbc hl,bc	; subtract column from value
	jr c,savdig	; save digit if we've gone negative
	inc a		; otherwise increase digit counter
	jr tstdig	; subtract again
.savdig	add hl,bc	; pull our subject value back out of the negative
	add a,$30	; turn counter into an ASCII digit
	ld (de),a	; save this digit
	inc de		; prepare to save next digit, if called again
	ret		; all done - return

	;
	; Choose mission routine
	;

	; input mission number and code
.chsemn	ld hl,newmis	; point to mission/code message
	call msgprt	; print it
	ld d,0		; zero upper byte of cursor offset
	ld e,10		; point cursor to character 10 of message
.chloop	ld hl,$5ac4	; point to message area attribute
	add hl,de	; advance to attribute under cursor
	ld (hl),$78	; make it bright inverse white
.ch_act	call ctlrel	; ensure no controls are activated
	call ctlact	; wait for a control to be activated
	bit 4,a		; is fire activated?
	jr nz,chdone	; yes, done
	bit 3,a		; is up activated?
	jr nz,ch__up	; yes, increase digit
	bit 2,a		; is down activated?
	jr nz,chdown	; yes, decrease digit
	bit 1,a		; is left activated?
	jr nz,chleft	; yes, move left
	; move cursor right
	ld (hl),$44	; unhighlight cursor
	inc e		; move cursor right
	ld a,13		; check the last level digit
	cp e		; have we moved past it?
	jr nz,ch_end	; no, check for end
	ld e,20		; yes, move to first digit of code
.ch_end	ld a,22		; check last code digit
	cp e		; have we moved past it?
	jr nz,chloop	; no, back to loop
	ld e,10		; yes, back to first character
	jr chloop	; back to loop
	; move cursor left
.chleft	ld (hl),$44	; unhighlight cursor
	dec e		; move cursor left
	ld a,9		; check first level digit
	cp e		; have we moved past it?
	jr nz,chcode	; no, check first digit of code instead
	ld e,21		; yes, point to last digit of code
.chcode	ld a,19		; check first code digit
	cp e		; have we moved past it?
	jr nz,chloop	; no, back to loop
	ld e,12		; yes, move to last digit of level
	jr chloop	; back to loop
	; decrease digit under cursor
.chdown	ld hl,newmis	; point to new mission level/code message
	add hl,de	; advance to cursor position
	dec (hl)	; decrement the digit
	ld a,$2f	; check for the ASCII code before "0"
	cp (hl)		; have we reached it?
	jr nz,chupdm	; no, so on to update the message
	ld (hl),$39	; cycle the digit back to "9"
	jr chupdm	; on to update the message
	; increase digit under cursor
.ch__up	ld hl,newmis	; point to the new mission level/code message
	add hl,de	; advance to cursor position
	inc (hl)	; increment the digit
	ld a,$3a	; check for ASCII code after "9"
	cp (hl)		; have we reached it?
	jr nz,chupdm	; no, go on to update the message
	ld (hl),$30	; cycle the digit back to 0
	; update the message after altering a digit
.chupdm	ld hl,newmis	; point to the mission level/code message
	call msgprt	; print it
	jr chloop	; back to loop
	; done choosing - calculate mission and code numbers
.chdone	ld (hl),$44	; remove white cursor from message
	ld hl,0		; zero level calculation
	ld a,(newmis+10); check hundreds digit
	and $0f		; reduce to 0..9
	call ch_add	; multiply HL by 10 and add A
	ld a,(newmis+11); check tens digit
	and $0f		; reduce to 0..9
	call ch_add	; multiply HL by 10 and add A
	ld a,(newmis+12); check units digit
	and $0f		; reduce to 0..9
	call ch_add	; multiply HL by 10 and add A
	ex de,hl	; put mission number aside
	ld hl,0		; zero code calculation
	ld a,(newmis+20); check tens digit
	and $0f		; reduce to 0..9
	call ch_add	; append to code
	ld a,(newmis+21); check tens digit
	and $0f		; reduce to 0..9
	call ch_add	; append to code
	ex de,hl	; swap codes
	; check the level code
	call gncode	; generate the actual code for the proposed level
	cp e		; is it the same as the one entered?
	jr nz,ch_err	; no, so complain
	; go to desired mission
	ld (mssion),hl	; store new mission
	jp newlvl	; generate new level
	; erroneuos code entered
.ch_err	ld hl,incode	; point to "invalid code" message
	call notprt	; print it
	ld hl,(mssion)	; check current mission
	call gncode	; regenerate its correct code (is this necessary?)
	jp maingm	; back to main game (temporary)

	; append a digit to the end of a number (like calculator input)
	; inputs:
	;    A - the digit to append
	;    HL - the value to append the digit to
	; outputs:
	;    HL - the updated value
.ch_add	push de		; store DE before corrupting it
	add hl,hl	; multiply by 10
	push hl		;
	pop de		;
	add hl,hl	;
	add hl,hl	;
	add hl,de	;
	ld d,0		; transfer A to DE
	ld e,a		;
	add hl,de	; add digit to HL
	pop de		; restore uncorrupted DE
	ret		; all done

	;
	; New Game Routine
	;

	; get confirmation and start new game
.ngconf	ld hl,sgcnfm	; point to confirmation message
	call msgprt	; print it
	call ctlact	; wait for control to be activated
	bit 4,a		; was fire activated?
	jp nz,newgam	; yes, start new game
	jp maingm	; otherwise, continue game
	
	;
	; messages
	;

	; confirmation messages
.slcnfm	defm "FIRE TO RESTART MISSION."
.nlcnfm	defm " FIRE FOR NEXT MISSION. "
.sgcnfm	defm "  FIRE TO RESTART GAME  "

	; notification messages
.gmover	defm " ALL MISSIONS COMPLETED "
.incmsg	defm "   MISSION INCOMPLETE   "
.newmis	defm "  MISSION XXX: CODE XX  "
.incode	defm "  INVALID MISSION CODE  "
