	;
	; Ossuary - an adventure game for the Sinclair ZX Spectrum
	; Graphics output routines and data
	;
	; Contains the code for drawing tiles on the map and inventory
	; screens, and the data for all the tiles.
	;
	; Copyright (C) Damian Walker 2013
	; Created 28-Jan-2013
	;

	;
	; Map graphics routines and data
	;

	; calculate location of top-left of window
	; inputs:
	;   (hero_l) = location of hero
	;   (mapwin) = location at top-left of visible window
.mapclc	push bc		; store registers to be corrupted
	ld a,(hero_l)	; what is hero's location?
	ld b,a		; store temporarily
	and $0f		; isolate column
	ld c,a		; store that in C
	ld a,b		; retrieve full coordinates
	and $f0		; isolate row number
	cp $50		; is row above 5?
	jr c,map_r0	; yes, so 0 is the top visible row
	sub $40		; otherwise calculate the top visible row
	cp $80		; is top row below 7?
	jr c,map_rs	; no, so go to store row
	ld a,$70	; lowest top row is 7
	jr map_rs	; go to store row
.map_r0	xor a		; top row is zero
.map_rs	ld b,a		; store the row temporarily
	ld a,c		; now check the column
	cp $05		; is column to the left of 5?
	jr c,map_c0	; yes, so 0 is the leftmost visible column
	sub $04		; otherwise calculate the leftmost visible column
	cp $08		; is the column to the right of 7?
	jr c,map_cs	; no, so on to store column and row
	ld a,$07	; rightmost leftmost column (!) is 7
	jr map_cs	; go to store column
.map_c0	xor a		; leftmost column is zero
.map_cs	or b		; combine row and column
	ld (mapwin),a	; store the top left window coordinate
	pop bc		; restore corrupted registers
	ret		; all done

	; draw the visible map window
.mapdrw	push bc		; store registers to be corrupted
	di		; this needs to be fast, used for cheap scrolling
	ld b,$09	; there are nine rows, initialise counter
	ld c,$00	; start at the top row
.mapdrr	call maprow	; display the map row
	inc c		; signify the next row
	djnz mapdrr	; and display it
	ei		; enable interrupts as we use them for timing
	pop bc		; restore values of registers corrupted
	ret		; done

	; draw a single row of the map window
	; inputs:
	;   C = row of map window to draw
.maprow	push bc		; store registers about to be corrupted
	push de		;
	push hl		;
	push ix		;
	ld ix,mapscr	; point to map screen row table
	ld b,0		; upper half of map row screen address offset
	ld a,c		; transfer row for arithmetic
	add a,a		; shift row into upper half of byte ...
	add a,a		;
	ld c,a		; ... while storing 4*row in (B)C ...
	add ix,bc	; ... to add to screen address pointer
	add a,a		;
	add a,a		;
	ld hl,mapwin	; point to MAPWIN variable (top-left location)
	add a,(hl)	; add this to our row number, giving leftmost row
	ld c,a		; turn this into a 16-bit offset (B still = 0)
	ld hl,lvlmap	; point HL to the level map
	add hl,bc	; add our offset, hl points to leftmost location
	ld b,9		; there are nine columns per row
.mapr_0	ld a,(hl)	; what is at this location?
	call maptil	; what tile should be shown?
	ld c,a		; store this for later reference
	ld a,(ix+$00)	; leftmost tile should be column 7 on screen
	add a,$12	; add width of map window
	sub b		; subtract twice the column counter
	sub b		;
	ld e,a
	ld d,(ix+$01)	; point to correct screen pixel row
	ld a,c		; retrieve tile number
	call drhalf	; draw the top half of the tile
	ld a,(ix+$02)	; leftmost tile should be column 7 on screen
	add a,$12	; add width of map window
	sub b		; subtract twice the column counter
	sub b		;
	ld e,a
	ld d,(ix+$03)	; point to correct screen pixel row
	ld a,c		; retrieve tile number
	or $80		; draw the bottom half of the tile
	call drhalf	;
	inc hl		; point HL to next map location
	djnz mapr_0	; continue till all nine tiles shown
	pop ix		; restore corrupted registers
	pop hl		;
	pop de		;
	pop bc		;
	ret		; all done

	; Screen addresses for each map row (upper byte)
.mapscr	defw $4027,$4047,$4067,$4087,$40a7,$40c7
	defw $40e7,$4807,$4827,$4847,$4867,$4887
	defw $48a7,$48c7,$48e7,$5007,$5027,$5047
	; Display variables
.mapwin	defs 1

	; Calculate map tile for a given block contents
	; inputs:
	;   A = contents of map block
	; outputs:
	;   A = tile to display
.maptil	cp $10		; is it a wall tile?
	ret c		; yes, wall tiles stored as tile number
	jr nz,mapcit	; not a floor either, on to check for items
	ld a,$04	; this is an open floor tile
	ret		; return
.mapcit	cp $20		; is this something more than an item?
	jr nc,mapher	; yes, so check for hero
	cp $1f		; is it the exit?
	jr z,mapexi	; yes, so process that
	ret		; return
.mapexi	ld a,$05	; this is the stairwell
	ret		; return
.mapher	cp $30		; is this something other than a hero?
	jr nc,mapnec	; yes, so check for the necromancer
	ld a,$06	; 6 is the hero graphic
	ret		; return it
.mapnec	cp $40		; is this something other than the necromancer?
	jr nc,mapnme	; yes, check for enemy
	ld a,$07	; 7 is the necromancer graphic
	ret		; return it
.mapnme	srl a		; rotate bits so A contains tile for enemy
	srl a		;
	srl a		;
	srl a		;
	ret		; return

	;
	; Panel Display
	;

	; draw the stats
.drstat	push bc		; store registers about to be corrupted
	ld bc,$0300	; there are three modified stats, starting at strength
.drst_0	call drsing	; display it
	call dr_mod	; display the modifier
	inc c		; advance to next stat
	djnz drst_0	; repeat until all modified stats shown
	call drsing	; display it
	call dr_max	; display maximum health
	pop bc		; retrieve corrupted registers
	ret		; all done

	; draw a single stat
	; inputs:
	;   C = stat number
.drsing	push bc		; store registers about to be corrupted
	push de		;
	push hl		;
	ld hl,hero_s	; point to hero strength
	ld d,0		; load stat number into DE
	ld e,c		;
	add hl,de	; advance pointer to correct stat
	ld l,(hl)	; load stat from pointer
	ld h,0		; zero MSB of stat
	ld de,statdt	; point to the stat digits
	push bc		; store (B)C for later retrieval
	ld bc,10	; calculate the 10s
	call digit	;
	pop bc		; retrieve (B)C
	ld a,l		; calculate the 1s
	add a,$30	;
	ld (de),a	;
	ld b,2		; display a 2-digit number
	ld hl,statsc	; point to stat screen addresses table
	ld e,c		; load DE with stat number
	ld d,0		;
	add hl,de	; advance pointer to correct screen address
	add hl,de	;
	ld e,(hl)	; point DE to the screen address
	inc hl		;
	ld d,(hl)	;
	ld hl,statdt	; point to the stat digits
	call msgprc	; print them
	pop hl		; retrieve corrupted registers
	pop de		;
	pop bc		;
	ret		; all done
	; data
.statsc	defw $4023,$4063,$40a3,$40e3 ; screen addresses for stats
.statdt	defs 2		; digits for the statistic

	; draw a single modifier
.dr_mod	push bc		; store registers about to be corrupted
	push de		;
	push hl		;
	ld hl,heroms	; point to hero strength
	ld d,0		; load stat number into DE
	ld e,c		;
	add hl,de	; advance pointer to correct stat
	ld a,(hl)	; load stat from pointer
	ld de,mod_dt+1	; point to the stat digits
	add a,$30	;
	ld (de),a	;
	ld b,2		; display a 2-digit number
	ld hl,mod_sc	; point to stat screen addresses table
	ld e,c		; load DE with stat number
	ld d,0		;
	add hl,de	; advance pointer to correct screen address
	add hl,de	;
	ld e,(hl)	; point DE to the screen address
	inc hl		;
	ld d,(hl)	;
	ld hl,mod_dt	; point to the stat digits
	call msgprc	; print them
	pop hl		; retrieve corrupted registers
	pop de		;
	pop bc		;
	ret		; all done
	; data
.mod_sc	defw $4043,$4083,$40c3,$4803 ; screen addresses for stats
.mod_dt	defm "+0"	; digits for the statistic


	; draw maximum health - AND colour current/max appropriately
.dr_max	push bc		; store registers about to be corrupted
	push de		;
	push hl		;
	xor a		; set running total to zero
	ld hl,hero_s	; point to strength
	add a,(hl)	; add strength
	inc hl		; point to agility
	add a,(hl)	; add agility
	inc hl		; point to intelligence
	add a,(hl)	; add intelligence
	ld l,a		; turn into 16-bit value
	ld h,0		;
	ld de,max_dt	; point to max health digits
	push de		; store for later
	ld bc,10	; calculate the 10s
	call digit	; 
	ld a,l		; calculate the 1s
	add a,$30	;
	ld (de),a	;
	ld b,2		; this is a two-digit number
	ld de,$4803	; point to health maximum
	pop hl		; retrieve pointer to digits
	push de		; store screen pointer
	call msgprc	; display the value
	pop hl		; retrieve pointer to screen characters
	ld (hl),$7f	; display fraction bar
	inc hl		;
	ld (hl),$fe	;
	; colour the health indicator in the appropriate colour
	ld a,(hero_h)	; what is the hero's health?
	cp 10		; is 10 greater than the hero's health?
	jr c,hl_red	; yes, so colour the health red
	ld a,$47	; otherwise start with bright white on black
	jr hl_clr	; on to colour it in
.hl_red	ld a,$42	; start with bright red on black
.hl_clr	ld hl,$58e3	; point to current health indicator attribute
	ld (hl),a	; colour in the first character ...
	inc hl		; ... and the second
	ld (hl),a	; 
	and $3f		; remove bright bit
	ld hl,$5903	; move on to maximum health indicator
	ld (hl),a	; colour in the first character ...
	inc hl		; ... and the second
	ld (hl),a	; 
	pop hl		; restore values of corrupted registers
	pop de		;
	pop bc		;
	ret		; all done
	; data
.max_dt	defm "00"	; digits for the statistic

	; draw the gold counter
.drgold	push bc		; store registers about to be corrupted
	push de		;
	push hl		;
	ld hl,(hergld)	; see how much gold we have
	ld de,golddt	; point to digits of gold
	;ld bc,1000	; calculate the 1000s
	;call digit	;
	ld bc,100	; calculate the 100s
	call digit	;
	ld bc,10	; calculate the 10s
	call digit	;
	ld a,l		; calculate the 1s
	add a,$30	;
	ld (de),a	;
	ld b,4		; display a 4-digit number
	ld de,$4841	; point to screen address
	ld hl,golddt	; point to numeric digits
	call msgprc	; print the characters
	pop hl		; retrieve registers that were corrupted
	pop de		;
	pop bc		;
	ret		; all done
.golddt	defm "0000"	; last digit will always be 0

	; draw the inventory
.dr_inv	push bc		; store registers about to be corrupted
	push de		;
	push hl		;
	ld b,$0c	; there are 12 inventory slots
	ld ix,inv_sc	; point to screen position table
	ld hl,herinv	; point to the hero's inventory
.diloop	ld e,(ix+$00)	; load screen address into DE
	ld d,(ix+$01)	;
	inc ix		;
	inc ix		;
	ld a,(hl)	; what's in the inventory?
	add a,$10	; this is the tile number
	ld c,a		; remember tile number
	push de		; remember screen address
	call drhalf	; draw upper half of tile
	pop de		; recall screen address
	ld a,e		; move one line down
	add $20		;
	ld e,a		;
	ld a,c		; recall sprite number
	or $80		; set bit for lower half
	call drhalf	; draw lower half of sprite
.dinext	inc hl		; look at next inventory item
	djnz diloop	; continue till all inventory done
	pop hl		; recall values of corrupted registers
	pop de		;
	pop bc		;
	ret		; all done

	; highlight an inventory slot
	; inputs:
	;   B = paper colour attribute (00ppp000)
	;   C = item number
	; outputs:
	;   C = item number preserved
.hi_inv	push hl		;
	push ix		;
	ld ix,inv_at	; point to attribute address table
	xor a		; zero the accumulator
.hi_i_0	cp c		; is B zero also?
	jr z,hi_i_1	; yes, so IX contains the correct attribute address
	inc ix		; point to next attribute address
	inc ix		;
	inc a		; count through the objects
	jr hi_i_0	; continue till IX contains correct attribute address
.hi_i_1	ld h,(ix+$01)	; load high byte, and ...
	ld l,(ix+$00)	; ... low byte of the address
	call setpap	; set the paper colour as directed
	inc hl		; point to top right attribute
	call setpap	; set the paper colour
	ld a,l		; point to bottom left attribute
	add $1f		; 
	ld l,a		;
	call setpap	; set the paper colour
	inc hl		; point to bottom right attribute
	call setpap	; set the paper colour
	pop ix		; retrieve uncorrupted values of registers
	pop hl		;
	ret		; done

	; turn a parciular attribute square the appropriate paper colour
	; inputs:
	;   B = paper colour attribute (00ppp000)
	;   HL = attribute location
.setpap	ld a,(hl)	; take corrent attribute value
	and $c7		; blacken the paper keeping ink, bright, flash
	or b		; apply the desired paper colour
	ld (hl),a	; save the modified value
	ret		; all done

	; data - screen positions
.inv_sc	defw $403b,$403d,$409b,$409d,$40db,$40dd
	defw $481b,$481d,$485b,$485d,$489b,$489d
	; data - attribute positions
.inv_at	defw $583b,$583d,$589b,$589d,$58db,$58dd
	defw $591b,$591d,$595b,$595d,$599b,$599d

	;
	; Tile Drawing Routines
	;

	; draw half a map graphic
	; inputs:
	;   A = hsssssss; h=top(0)/bottom(1); s=sprite number
	;   DE = screen address
.drhalf	push bc		; save registers to be overwritten
	push hl		;
	sla a		; shift top bit of A into carry
	rl c		; and store in bottom bit of c
	srl a		; restore A without its top bit
	ld b,a		; store a into counter
	ld hl,tiles	; point to start of map sprite graphics
	push de		; remember screen address DE for a moment
	bit 0,c		; which half of the sprite are we printing?
	jr z,findsp	; find the right sprite
	ld de,$0010	; bottom, so skip top half of sprite data ...
	add hl,de	; ... by adding it to HL	
.findsp	ld de,$0024	; set D to length of sprite information (de)
	and b		; is counter 0?
	jr z,sfound	; if so, we're already pointing at the right sprite
.nextsp	add hl,de	; otherwise point to next sprite ...
	djnz nextsp	; ... and check again
	; draw the pixels
.sfound	pop de		; retrieve screen address
	push de		; and store it again as we'll be corrupting it
	ld b,8		; 8 lines for this half of sprite
.pixrow	ldi		; transfer first byte from sprite to screen
	ldi		; transfer second byte from sprite to screen
	inc bc		; we didn't want to decrement BC...
	inc bc		; ... so restore it again
	dec e		; point back to left-hand character ...
	dec e		;
	inc d		; ... then point to next screen row
	djnz pixrow	; and output the next row
	; paint the attributes
	bit 0,c		; if zero, we're drawing the top half of the sprite
	jr nz,drattr	; if non-zero, we're drawing the bottom
	ld de,$000e	; we want to skip 14 bytes first
	add hl,de	;
.drattr	inc hl		; skip two more bytes to reach the correct attr source
	inc hl		;
	pop de		; regain screen location
	call pix2at	; convert pixel address to attribute address
	ldi		; transfer first attribute
	ldi		; transfer second attribute
	pop hl		; restore registers
	pop bc		;
	ret		; done with this half square

	; convert a pixel address to an attribute address
	; inputs:
	;   DE = pixel address
	; outputs:
	;   DE = atribute address
.pix2at	srl d		; shift D so that ...
	srl d		; ... 010TTPPP -> 000010TT ...
	srl d		; ... where T = third of screen & P = char pixel line
	ld a,$50	; prepare accumulator with 01010000 and OR so that ...
	or d		; ... 000010TT becomes 010110TT
	ld d,a		; DE now contains the attribute address
	ret		; all done

	;
	; Graphical Data
	;

	; ordinary wall
.tiles	defb $73,$fe,$d5,$55,$a6,$aa,$d3,$54,$aa,$a2,$d1,$1d,$a8,$6a,$46,$d4
	defb $1b,$aa,$74,$d4,$aa,$40,$d4,$7e,$aa,$ab,$d4,$d4,$aa,$aa,$40,$40
	defb $41,$41,$41,$41
	; wall with grotesque face
	defb $70,$3e,$01,$80,$4f,$f2,$7f,$fe,$31,$8c,$95,$a9,$9b,$da,$4e,$70
	defb $0f,$f2,$6c,$34,$a9,$90,$c7,$e6,$a3,$cb,$d1,$94,$a9,$aa,$40,$00
	defb $41,$41,$41,$41
	; wall with niche
	defb $70,$3e,$c1,$85,$ad,$b2,$8c,$30,$b0,$8c,$b0,$4d,$01,$20,$60,$56
	defb $60,$b6,$00,$00,$7b,$de,$7b,$de,$00,$00,$d4,$d4,$aa,$aa,$40,$40
	defb $41,$41,$41,$41
	; wall with torch
	defb $72,$ff,$d5,$ff,$a7,$ff,$d3,$f7,$ab,$ef,$d1,$e7,$a8,$e7,$46,$ff
	defb $1b,$82,$74,$b4,$aa,$30,$d4,$66,$aa,$4b,$d4,$54,$aa,$0a,$40,$40
	defb $41,$71,$41,$41
	; floor
	defb $00,$00,$00,$40,$20,$02,$00,$00,$04,$08,$00,$00,$20,$20,$01,$04
	defb $00,$00,$40,$10,$04,$00,$00,$00,$00,$00,$20,$80,$00,$14,$00,$00
	defb $06,$06,$06,$06
	; staircase down
	defb $0d,$b0,$6d,$b6,$60,$06,$00,$00,$c8,$03,$d9,$03,$1a,$20,$db,$4b
	defb $db,$23,$1b,$68,$db,$63,$db,$6b,$1b,$60,$60,$06,$6d,$b6,$0d,$b0
	defb $05,$05,$05,$05
	; the hero
	defb $38,$08,$54,$08,$7c,$18,$38,$10,$7c,$30,$fe,$20,$bd,$60,$d9,$40
	defb $39,$e0,$7c,$c0,$7c,$80,$66,$00,$36,$00,$36,$00,$36,$00,$77,$00
	defb $06,$47,$04,$02
	; the necromancer
	defb $03,$80,$04,$c0,$04,$e0,$63,$c0,$b9,$80,$bf,$e0,$7f,$f8,$03,$dc
	defb $03,$4c,$07,$aa,$0f,$a4,$1a,$d0,$16,$d0,$2d,$d8,$1d,$e8,$06,$c0
	defb $45,$05,$45,$05
	; a giant bat hovering above the floor
	defb $02,$40,$07,$e0,$1d,$b8,$3e,$7c,$7f,$fe,$f3,$cf,$c2,$43,$84,$21
	defb $00,$00,$40,$10,$04,$00,$00,$00,$00,$00,$20,$80,$00,$14,$00,$00
	defb $47,$07,$06,$06
	; vermin - a giant rat
	defb $00,$00,$00,$00,$00,$00,$60,$0c,$1c,$32,$03,$c1,$00,$01,$00,$02
	defb $00,$3c,$04,$fe,$0d,$fe,$37,$fe,$3f,$fc,$0f,$fc,$1f,$18,$70,$30
	defb $02,$02,$45,$05
	; arachnid - a giant spider
	defb $01,$80,$03,$c0,$07,$e0,$37,$ec,$cb,$d3,$07,$e0,$35,$ac,$4f,$f2
	defb $85,$a1,$1b,$d8,$22,$44,$45,$a2,$48,$12,$88,$11,$04,$20,$08,$10
	defb $43,$03,$43,$03
	; serpent - a giant snake
	defb $00,$00,$00,$e0,$01,$50,$01,$38,$00,$7c,$00,$cc,$00,$0c,$00,$18
	defb $0e,$70,$23,$c4,$3c,$3c,$5f,$fa,$e3,$c7,$7c,$3e,$1f,$f8,$03,$c0
	defb $47,$44,$44,$04
	; cadaver - a zombie
	defb $01,$c0,$02,$a0,$03,$e0,$01,$40,$07,$e0,$0f,$f0,$3b,$d8,$11,$b0
	defb $01,$c0,$03,$e0,$03,$e0,$06,$60,$06,$c0,$06,$c0,$06,$c0,$0e,$e0
	defb $07,$07,$44,$04
	; skeleton
	defb $03,$80,$05,$40,$43,$80,$42,$80,$49,$20,$57,$d0,$51,$0e,$57,$df
	defb $e1,$1b,$47,$df,$43,$8e,$02,$80,$04,$40,$04,$40,$04,$40,$0c,$60
	defb $47,$07,$47,$07
	; spectre
	defb $03,$80,$05,$40,$07,$e0,$63,$c0,$f9,$80,$ff,$e0,$7f,$f8,$5f,$fc
	defb $0f,$c6,$0f,$c2,$07,$c2,$07,$e0,$03,$f0,$00,$f0,$00,$18,$00,$00
	defb $41,$01,$41,$01
	; daemon
	defb $00,$81,$00,$bd,$00,$5a,$14,$3c,$14,$24,$08,$18,$04,$7e,$04,$ff
	defb $02,$bd,$02,$bd,$01,$18,$01,$3c,$00,$3c,$00,$66,$00,$66,$00,$ee
	defb $45,$42,$45,$42
	; a blank tile - cheaper than a "blank routine" :-)
	defb $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	defb $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	defb $00,$00,$00,$00
	; a purse of gold
	defb $00,$00,$00,$20,$00,$11,$00,$12,$00,$22,$00,$4c,$06,$70,$0f,$c0
	defb $00,$f0,$0e,$7c,$00,$7e,$37,$7e,$00,$7f,$77,$7f,$00,$7e,$ee,$fc
	defb $02,$06,$46,$02
	; a casket of gold
	defb $00,$00,$00,$00,$7f,$e0,$40,$78,$20,$38,$1c,$1c,$36,$0c,$6d,$08
	defb $ff,$f8,$81,$f8,$99,$d8,$81,$f8,$81,$f8,$81,$f8,$81,$f8,$ff,$f8
	defb $46,$02,$16,$02
	; an apple
	defb $00,$86,$00,$9e,$00,$5c,$00,$7c,$00,$70,$00,$40,$00,$80,$0e,$80
	defb $1f,$30,$3f,$f8,$3f,$fc,$3f,$fc,$3f,$ec,$1f,$c8,$0f,$d8,$06,$70
	defb $06,$04,$42,$02
	; a potion
	defb $07,$e0,$02,$40,$02,$40,$02,$40,$02,$40,$04,$20,$08,$10,$08,$10
	defb $0f,$f0,$0f,$f0,$0f,$f0,$0f,$f0,$0f,$f0,$0f,$f0,$0f,$f0,$07,$e0
	defb $47,$07,$45,$05
	; a dagger
	defb $00,$00,$00,$00,$00,$00,$00,$00,$00,$10,$00,$60,$02,$e0,$03,$c0
	defb $03,$80,$07,$c0,$0e,$00,$0c,$00,$00,$00,$00,$00,$00,$00,$00,$00
	defb $04,$47,$02,$04
	; a sword
	defb $00,$01,$00,$07,$00,$0e,$00,$1c,$00,$38,$04,$70,$07,$e0,$03,$c0
	defb $03,$c0,$07,$c0,$0e,$60,$1c,$00,$18,$00,$00,$00,$00,$00,$00,$00
	defb $04,$47,$02,$04
	; an axe
	defb $01,$e0,$07,$e0,$07,$f0,$0f,$f8,$0f,$fc,$0f,$ff,$03,$ff,$01,$ff
	defb $03,$ff,$07,$7e,$0e,$3e,$1c,$38,$38,$00,$70,$00,$e0,$00,$c0,$00
	defb $47,$07,$02,$05
	; a wooden shield
	defb $00,$00,$00,$00,$03,$c0,$0d,$b0,$16,$68,$19,$98,$2b,$d4,$36,$6c
	defb $36,$6c,$2b,$d4,$19,$98,$16,$68,$0d,$b0,$03,$c0,$00,$00,$00,$00
	defb $42,$02,$42,$02
	; an iron shield
	defb $00,$00,$03,$c0,$0e,$70,$17,$e8,$2f,$f4,$3f,$fc,$7e,$7e,$5d,$ba
	defb $5d,$ba,$7e,$7e,$3f,$fc,$2f,$f4,$17,$e8,$0e,$70,$03,$c0,$00,$00
	defb $47,$07,$47,$07
	; a scroll
	defb $70,$3c,$ff,$fa,$8f,$c7,$10,$3f,$3f,$ff,$24,$4f,$7f,$fe,$61,$46
	defb $7f,$fc,$68,$9c,$3f,$f6,$3f,$fe,$38,$1e,$c7,$ee,$ff,$f4,$70,$38
	defb $06,$06,$06,$06
	; an amulet
	defb $00,$00,$14,$00,$2a,$a0,$41,$50,$20,$08,$10,$04,$20,$02,$10,$04
	defb $0c,$08,$1e,$04,$3f,$08,$39,$44,$1a,$a8,$0c,$10,$00,$00,$00,$00
	defb $46,$46,$04,$46
	; a wand
	defb $00,$00,$00,$00,$00,$70,$01,$f8,$02,$fc,$02,$f4,$03,$f4,$03,$78
	defb $03,$c8,$07,$f0,$0e,$00,$1c,$00,$38,$00,$70,$00,$e0,$00,$c0,$00
	defb $46,$04,$02,$06
	;defb $00,$00,$00,$00,$00,$30,$01,$78,$02,$fc,$02,$f4,$03,$68,$03,$30
	;defb $03,$c8,$07,$f0,$0e,$00,$1c,$00,$38,$00,$70,$00,$e0,$00,$c0,$00
	;defb $46,$04,$02,$06
	;defb $00,$00,$00,$00,$01,$80,$0d,$b0,$07,$e0,$03,$c0,$07,$e0,$0e,$70
	;defb $01,$80,$01,$80,$01,$80,$01,$80,$01,$80,$01,$80,$01,$80,$01,$80
	;defb $45,$05,$42,$02
