;	=======================================================================
;
;	Intergalactic Space Rescue - a puzzle game for the 16K ZX Spectrum
;	Assembler code for the game program
;
;	Copyright (C) Damian Walker 2012
;	Created 10-Dec-2012
; 
;	=======================================================================
;
;	ZX81 port by GZS (Gaal, Zsolt)                             - 07/04/2013
;	ZXPAND joystick interface by Charlie Robson (sirmorris)    - 12/04/2013
;	ZONX-81 compatible sounds by Kelly Abrantes Murta (kmurta) - 15/04/2013
;
;	=======================================================================
;
	.org $4009			; 16393

	.db 0				; VERSN
	.dw $0000			; E_PPC
	.dw $0000			; D_FILE
	.dw $0000			; DG_CC
	.dw Variables			; VARS
	.dw $0000			; DEST
	.dw eof_Vars			; E_LINE
	.dw eof_Vars+4			; CH_ADD
	.dw $0000			; X_PTR
	.dw $0000			; STKBOT
	.dw $0000			; STKEND
	.db 0				; BREG
	.dw $0000			; MEM
	.db 0				; UNUSED1
	.db 0				; DG_SZ
	.dw $0000			; S_TOP
LAST_K	.dw $FFFF			; 
	.db 0				; DEBOUN
MARGIN	.db 0				; MARGIN
	.dw Begin			; NXTLIN  -  autostart
	.dw 0				; OLDPPC
	.db 0				; FLAGX
	.dw $0000			; STRLEN
	.dw $0000			; T_ADDR
	.dw $0000			; SEED

FRAMES	.dw $FFFF			; 

	.dw $0000			; COORDS
	.db $0000			; PR_CC
	.dw $0000			; S_POSN
	.db $40				; CDFLAG
;
;       =======================================================================
;
DF_ECHO	.equ $+$8000			; upper 32K
	;
	ld r,a				; low byte of rowaddr.
lbuf
	.db $40,$40,$40,$40,$40,$40	; hide 6 columns (left)
	.db $80,$80,$80,$80,$80,$80,$80,$80,$80,$80
	.db $80,$80,$80,$80,$80,$80,$80,$80,$80,$80
	.db $40,$40,$40,$40,$40,$40	; hide 6 columns (right)
	;
	jp (ix)				; (return from DFILE echo)
;
;	========================================================================
HFILE					; @ $4060
	.incbin "ZX81_title_scr.obj"	; the linearized title screen
	.org $5860			; without attribute field (6KB)
;
; ==============================================================================
; THE DISPLAY GENERATOR
;	========================================================================
DG_INIT
	ld hl,HFILE-$20		; hl points to the linear screen map
	ld de,$00C1		; 192 lines (+1)
	ld bc,$051F		; delay (5*13+8=T73)
DG_INIT1
	ld ix,DG_CONT		; DFILE echo return addr. 
	;
	djnz $			; delay
;
; ==============================================================================
;	Executing D-File cont. 
DG_CONT
	dec e			;

DG_QUIT	.equ $+1		;

	jp z,DG_EXIT		;
	nop			; delay
	inc hl			; correct the offset
	add hl,bc		; address of the next row
	ld a,h			; high byte of rowaddr.
	ld i,a			; to "I" reg.
	ld a,l			; low byte of rowaddr. ..
	jp DF_ECHO		; (execute DFILE echo)
;
;	ld r,a			; .. to "R" reg.
;	32x NOP			; the dummy DFILE
;	jp (ix)			; (return from DFILE echo)
; 
; ==============================================================================
;	End of displaytask: 
DG_EXIT
	dec (IY+$28)		; set sv. MARGIN ($4028)

	call $0292		; return to application program

	ld b,10			; delay
	djnz $			; 

	call $0220		; extra register PUSH and VSYNC

DG_ISET	.equ $+2		;

	ld ix,DX_INIT		; reload the hr vector
	jp $02A4		; return to application program
; 
; ==============================================================================
; THE modified DISPLAY GENERATOR (for the title screen)
;	========================================================================
DX_INIT
	ld hl,DX_CONT		; 
	ld (DG_QUIT),hl		; 
	;
	ld hl,HFILE+$00E0	; skip the first 8 scan lines
	ld bc,$811F		; using delay (128*13+8 TS)
	ld de,$00A0		; then display 160 scan lines
	;
	inc de			; 
	jr DG_INIT1		; 
DX_CONT
	ld hl,DX_CONT1		; 
	ld (DG_QUIT),hl		; 
	;
	ld hl,DG_INIT-$0220	; skip the next 8 scan lines
	ld bc,$781F		; using delay (119*13+8 TS)
	ld e,$08		; then display 8 scan lines
	;
	inc e			;
	nop			; 
	jr DG_INIT1		; 
DX_CONT1
	ld hl,DG_EXIT		; 
	ld (DG_QUIT),hl		; 
	;
	ld bc,$781F		; skip the last 8 scan lines
	djnz $			; delay (119*13+8 TS)
	;
	jr DX_CONT2		; delay
DX_CONT2
	jr DG_EXIT		; 
;
; ==============================================================================
; START (init)  -  THE TITLE SCREEN
;	========================================================================
INIT
	call $0207		; switch the SLOW mode
	ld ix,DX_INIT		; reload the hr vector
;
;	-------------------------------------------------------------
;	AY initialization
;
	xor a		; set chanel A freq to zero
	out ($df),a	;
	out ($0f),a	;
	inc a		;
	out ($df),a	;
	xor a		;
	out ($0f),a	;
	ld a,$08	; set volume in chanel A to 15
	out ($df),a	;
	ld a,$0f	;
	out ($0f),a	;
	ld a,$07	; select the mixer AY reg
	out ($df),a	;
	ld a,$ff	;set off all chanels
	out ($0f),a     ;(the reg 7 continues addressed for future accesses)
;
;	-------------------------------------------------------------
;	Instruct ZXpand to latch the joystick data,
;	probably not necessary but it won't hurt
;
	ld bc,$e007	;
	ld a,$a0	;
	out (c),a	;
;
;	-------------------------------------------------------------
;	Ascertain the control method
;
	; get the player's response
ctlget
	ld b,$19	; timing counter for flashing
ctlget0
	call waitfrm	; VSYNC and KBD-read in background
	;
	ld a,(LAST_K)	; let's look at row
	cp $bf		; H-ENTER of keyboard
	jr nz,ctlget1	;

	ld a,(LAST_K+1)	; and the columns
	and $18		; isolate J and K
	cp $18		; are both keys up?
	jr nz,ctlset	; 
ctlget1
	djnz ctlget0	; flashing?
	;
	ld hl,$5668	; (K)
	call chpatch	; toggle
	;
	ld hl,$5670	; (J)
	call chpatch	; toggle
	;
	jr ctlget	; continue to read keyboard till J/K pressed

	; set the correct input routine for ctlpol
ctlset	and $08		; look specifically at the K key
	jr nz,ctlset1	; if K not pressed then we want joystick
	ld (ctlpol+1),a	; else use keyboard routine
ctlset1	 	
	ld hl,DG_INIT		; reload the hr vector
	ld (DG_ISET),hl		;
	;
	ld hl,lbuf+6		; Complete the
	ld de,lbuf+5		; left side
	ld bc,$0006		; of the
	lddr			; dummy DFILE
	ld e,(lbuf+26)&255 	; then the right
	ld c,$06		; side of the
	ldir			; dummy DFILE
;
;	-------------------------------------------------------------
clr_scr
	ld hl,HFILE+$106	; clear
	ld de,HFILE+$107	; the
	ld bc,$1F		; first
	ld (hl),0		; line of
	ldir			; the map

	ld hl,HFILE+$106	; and then
	ld de,HFILE+$126	; the next
	ld a,$9F		; 159 lines
clr_row
	ld c,20			; 20 bytes in
	ldir			; each lines

	ld c,12			; offset
	ex de,hl		; count the
	add hl,bc		; next position
	ex de,hl		; count the
	add hl,bc		; next position
	dec a			; set row counter
	jr nz,clr_row		; done?
;
; ==============================================================================
;	THE MAIN PROGRAM
;	========================================================================
;
;	declarations and main routines for the game
;
;	This looks more complicated than it would otherwise be because
;	I want to store data overwriting the now unused loader code, and the
;	game initialisation code once it has been executed.
;
;	The origin defines the start of the on-off initialisation routine.
;	Thus values written during initialisation must be placed BEFORE the
;	origin, and values written once initialisation is complete may be
;	placed AFTER the origin.
;
	;
	; Constants
	;

ctlopt = DX_INIT	; control option saved here 

	; display constants

msgpos	= $5664		; address of message display line 1st char top
msglen	= $18		; 24 characters in display window

	;
	; Addresses of variables in memory
	;

	; player control
menopt	= ctlopt + 1	; menu option chosen

	; random number generation
randsd	= menopt + 1	; the full random number seed
randsl	= randsd	; low byte of RANDSD
randsh	= randsd + 1	; high byte of RANDSD

	; game variables
mssion	= randsd + 2	; mission number
rescxy	= mssion + 2	; rescue ship position on map
beacxy	= rescxy + 1	; beacon position on map
shfuel	= beacxy + 1	; fuel left in ship
mapdat	= shfuel + 1	; 100 bytes of map data
rscini	= mapdat + 100	; initial position of rescue ship
rescdr	= rscini + 1	; last direction of rescue ship
lvcode	= rescdr + 1	; level code
psignl	= lvcode + 1	; previous signal strength ($5901)

;	======================================================================
;
;	Initialisation for game and for levels
;
	; create a new game
newgam	ld hl,$00	; current level number should be 0
	ld (mssion),hl	; initialise it

	; create a new level
newlvl	call lvlgen	; generate level
	call dr_map	; draw the map
	call drstat	; display fuel, beacon and location
	call mssprt	; display mission information
;
;	Main control routine
;
	; display status and wait for player to do something
maingm	call drstat	; display fuel, beacon and location
	call ctlrel	; ensure no controls are pressed
	call ctlact	; wait for a control to be pressed

	; act upon player's control selection
	bit 4,a		; is fire pressed?
	jp nz,menu	; yes, bring up the menu
	ld a,(rescxy)	; otherwise look at ship location
	ld d,a		; transfer it to D
	ld e,$0a	; take the map width
	call div8	; now D=Y and E=X

	; check we're not moving off the map
	ld hl,movtbl	; refer to the movement table
	ld a,(ctlopt)	; recall controls
	bit 0,a		; are we moving right?
	jr z,movecl	; no, check left
	ld a,9		; look at the right-hand edge
	cp e		; is that where we are?
	jr z,maingm	; yes, so don't move further
	jr movesh	; otherwise move ship
movecl	bit 1,a		; are we moving left?
	jr z,movecd	; no, check down instead
	ld a,e		; are we at the left edge?
	or a		;
	jr z,maingm	; yes, so don't move further
	jr movelt	; otherwise move ship
movecd	bit 2,a		; are we moving down?
	jr z,movecu	; no, check up instead
	ld a,9		; look at bottom edge
	cp d		; is that where we are?
	jr z,maingm	; yes, so don't move further
	jr movedn	; otherwise move ship
movecu	ld a,d		; are we at the upper edge?
	or a		;
	jr z,maingm	; yes, so don't move further

	; point HL to row of movement table for direction pressed
moveup	inc hl		; advance down the table thrice for up ...
movedn	inc hl		; ... twice for down ...
movelt	inc hl		; ... just once for left

	; check that we can and should move the rescue ship now
movesh	ld a,(shfuel)	; how much fuel to we have?
	or a		; is there any left?
	jr z,maingm	; no, so we can't move anyway
	ld a,(rescxy)	; recall ship position
	ld c,a		; set aside for a moment
	ld a,(beacxy)	; check beacon position
	cp c		; have we found the beacon?
	jr z,maingm	; yes, so there is no need to move

	; store the current signal strength
	call signal	; calculate the current signal
	ld (psignl),a	; store it in the "previous signal" variable

	; move the ship according to table entry HL
	ld a,c		; recall rescue ship position
	call drtile	; draw map contents at ship position
	ld a,(hl)	; get facing offset
	ld (rescdr),a	; store for drawing later
	add a,c		; add the cursor location to the facing offset
	call shipck	; check for effects on the ship
	jr maingm	; back to the game loop

	; movement table
movtbl	.db $01 ; movement right
	.db $ff ; movement left
	.db $0a ; movement down
	.db $f6 ; movement up
;
;	======================================================================
;
;	The menu system
;
	; entry point for the menu routine - decide default option
menu	ld a,(rescxy)	; check rescue ship location
	ld c,a		; put aside for a moment
	ld a,(beacxy)	; check beacon location
	cp c		; are the two locations the same?
	jr z,mnextm	; yes, so on to next mission
	ld a,(shfuel)	; what fuel have we?
	or a		; have we any?
	jr z,mstart	; no, so "restart mission"
	xor a		; make default option "cancel"
	jr menurp	; on to process the menu

	; player is out of fuel - "restart mission" default option
mstart	ld a,$02	; make default option "restart mission"
	jr menurp	; on to process the menu

	; player has found the beacon - "new game" or "next mission"
mnextm	ld hl,(mssion)	; what is our current mission?
	ld bc,999	; 999 is the last mission
	sbc hl,bc	; is 999 our mission?
	jr z,mnewgm	; yes, so "new game" - there is no next mission
	ld a,$03	; make default option "next mission"
	jr menurp	; on to process the menu
mnewgm	ld a,$05	; make default option "new game"

	; menu loop - print the currently selected menu option
menurp	ld (menopt),a	; store accumulator as last used option
	ccf		; clear the carry flag prior to rotation
	rlca		; convert to an offset
	rlca		;
	rlca		;
	ld e,a		;
	rlca		;
	add a,e		;
	ld e,a		;
	ld d,0		;
	ld hl,mentxt	; point to menu text table
	add hl,de	; point down to specific option's text
	call msgprt	; print the option text

	; wait for new LEFT/RIGHT activation or FIRE release
menctl	call ctlpol	; poll the control device
	and $0f		; just look at the directional bits
	jr nz,menctl	; repeat until all direction controls released
menupl	call ctlpol	; poll the control device
	bit 4,a		; has the fire button been released?
	jr z,menrel	; yes, process menu option
	and $03		; otherwise we're interested only in left/right
	jr z,menupl	; if neither is pressed, repeat

	; process LEFT/RIGHT control to cycle through menu options
	bit 1,a		; is left control activated?
	jr z,menrgt	; no, so we're moving right
	ld a,$ff	; otherwise we're moving left
menrgt	ld b,a		; store offset
	ld a,(menopt)	; look at last-chosen menu option
	add a,b		; add the offset
	cp $ff		; have we passed the first option?
	jr z,menust	; handle passing the first option elsewhere
	cp $06		; have we passed the last option?
	jr c,menurp	; no, so show new option
	xor a		; otherwise set to first option again ...
	jr menurp	; and show it

	; we've moved left past the first option - cycle to the last
menust	ld a,$05	; select last option on the menu
	jr menurp	; repeat menu

	; FIRE button released - act on this menu option
menrel	ld a,(menopt)	; retrieve menu option
	sla a		; convert it to an offset
	ld e,a		;
	ld d,0		;
	ld hl,menjmp	; point to the jump table
	add hl,de	; point to the specific option's entry
	ld e,(hl)	; load entry into DE
	inc hl		;
	ld d,(hl)	;
	ex de,hl	; swap entry into HL ...
	jp (hl)		; ... and jump there

	; table of menu option text
mentxt	.db "   CANCEL MENU OPTION   "
	.db "SHOW MISSION INFORMATION"
	.db "  RESTART THIS MISSION  "
	.db "   ON TO NEXT MISSION   "
	.db "    CHOOSE A MISSION    "
	.db "     BEGIN NEW GAME     "

	; table of jump locations for menu options
menjmp	.dw maingm	; cancel menu
	.dw msinfo	; show mission information
	.dw startm	; restart mission, after confirmation
	.dw nextmn	; start next mission, after confirmation
	.dw chsemn	; choose a mission
	.dw ngconf	; new game, after confirmation
;
;	======================================================================
;
;	Main generation routine
;
	; start by saving the registers
lvlgen	push bc		; preserve the register values
	push de		;
	push hl		;

	; display "Please wait" message
	ld hl,plwait	; point to "Please wait" message
	call notprt	; print it and impose an artificial delay

	; generate level code and initialise random seed
	ld hl,(mssion)	; look at mission number
	call gncode	; generate level code

	; fill the map with empty space
	ld hl,mapdat	; point to start of map
	ld (hl),0	; zero the first map square
	ld bc,99	; there are 99 more
	ld d,h		; point at the same map space
	ld e,l		;
	inc de		; one location further
	ldir		; copy the other 99 spaces

	; generate the planets
	ld b,10		; there are ten planets on every level
	ld c,1		; 1 is the code for a planet
	call featur	; generate the planets

	; generate the asteroids
	call diffic	; determine the number of asteroids on this level
	ld c,2		; 2 is the code for an asteroid
	call featur	; generate the asteroids

	; generate the hyperspaces
	call diffic	; determine the number of hyperspaces on this level
	ld c,3		; 3 is the code for a hyperspace
	call featur	; generate the hyperspaces

	; place the ships on the level
lships	call findmt	; find an empty space for the rescue ship
	ld a,e		; store it ...
	ld (rescxy),a	; ... as current position ...
	ld (rscini),a	; ... and initial position of rescue ship
	call findmt	; fina an empty space for the stricken ship
	ld a,e		; store it ...
	ld (beacxy),a	; ... as the beacon position
	call signal	; calculate signal strength
	cp 10		; is it 10 or stronger?
	jr nc,lships	; yes, we're too near, try again
	ld a,(rescxy)	; recall ship location
	call mapmem	; convert to an address
	set 7,(hl)	; mark location as "seen"
	ld a,$12	; get a tank full of fuel ...
	ld (shfuel),a	; ... and put it in the tank
	call signal	; calculate initial signal strength
	ld (psignl),a	; store in the "previous signal" variable

	; now randomise the number generator
	ld hl,(FRAMES)	; get random number from timer
	ld (randsd),hl	; save as next seed

	; all done
	pop hl		; restore the register values
	pop de		;
	pop bc		;
	ret		; done

	;
	; Subroutines
	;

	; generate code if level not 0
	; inputs:
	;   HL = level number
	; outputs:
	;   A = level code
	;   (lvcode) = level code
gncode	push de		; store DE before we corrupt it
	ld d,h		; create a number $0000..$ffff from level number ...
	ld e,l		;
	add hl,hl	;
	add hl,hl	;
	add hl,hl	;
	add hl,hl	;
	add hl,de	;
	ld (randsd),hl	; ... to use level number as random seed
	ex de,hl	; retrieve the level number
	ld a,h		; check if HL is 0
	or l		;
	jr z,svcode	; if it is, skip code calculation
	call rand16	; generate a random code
	ld a,(randsd)	; take LSB ...
	ld d,a		; ... as divisor
	ld e,100	; take 100 as dividend
	call div8	; do the division, remainder in D
	ld a,e		; take the remainder
svcode	ld (lvcode),a	; store the level code for later reference
	pop de		; restore DE
	ret		; return to calling process

	; generate map features
	; inputs:
	;   B = number of features to generate
	;   C = type of feature to generate
featur	push de		; preserve registers like a good boy
	push hl		;
featlp	call findmt	; find an empty map square
	ld (hl),c	; otherwise put map feature into square
	djnz featlp	; on to the next feature until done
	pop hl		; restore registers
	pop de		;
	ret		; all done

	; determine number of "difficult" map features for this level
	; outputs:
	;   B = number of features
diffic	push de		; preserve the registers like a good boy
	push hl		;
	ld hl,(mssion)	; what is the mission number?
	ld de,48	; we add an extra feature every 48 levels
	call div16	; (D)E now contains the quotient, 0..20
	ld a,10		; difficult features start at 10 on level 0
	add a,e		; add the number of extra features
	ld b,a		; put into B ready for use
	pop hl		; restore the register values
	pop de		;
	ret		; all done

	; find a random location that is currently empty
	; outputs:
	;  (D)E = map square location 0..99
	;   HL = map square address in memory
findmt	call rand16	; generate a random number
	ld hl,(randsd)	; look at it
	ld de,100	; we want it to be 0 to 99
	call div16	; HL holds the remainder, a random map square
	ex de,hl	; now DE is map square
	ld hl,mapdat	; point HL at start of map
	add hl,de	; advance HL to the desired map location
	ld a,(hl)	; what is there?
	or a		; is it clear?
	jr nz,findmt	; square occupied, so we need to look elsewhere
	ret

	;
	; Messages and other data
	;

	; messages
plwait	.db "     GENERATING MAP     "	;
;
; ============================================================================
;
; 	Gameplay routines
;

	; implement side effects on moving
	; inputs:
	;   A = proposed new ship location
	; outputs:
	;   A = actual new ship location
shipck	push bc		; store entry values of registers to be corrupted
	push de		;
	push hl		;
	ld (rescxy),a	; make this the ship location (for now)
	ld c,a		; put proposed ship location aside
	call mapmem	; get memory address for map location

	; process effects of ship location
nflash	ld a,(hl)	; what is there?
	and $7f		; mask out "explored" bit
	or a		; is it deep space?
	jr z,dspace	; yes, process it
	dec a		; planet?
	jr z,planet	; yes, process it
	dec a		; asteroids?
	jr z,asters	; yes, process them

	; to get here we must be at a hyperspace
	set 7,(hl)	; mark location as "seen"
	call h1anim	; show rotation animation
	ld a,(rescxy)	; recall location
	call drtile	; draw the tile as we're about to move away
hspace	call rand16	; get a random number 0..99
	ld hl,(randsd)	;
	ld de,100	;
	call div16	;
	ex de,hl	; put aside location number
	ld a,e		; what is the location number?
	call mapmem	; point at the address for that location
	ld a,(hl)	; what is there?
	and $7f		; mask out the the "seen" bit
	cp $03		; is it a hyperspace?
	jr nz,hspace	; it's not a hyperspace, try again
	ld a,e		; recall the location number
	cp c		; is it the same as the location we moved to?
	jr z,hspace	; yes, try again (hyperspace should go somewhere else)
	ld c,a		; make this our new proposed location
	ld (rescxy),a	;
	set 7,(hl)	; mark location as "seen"
	call h2anim	; secondary animation
	jr dspafl	; on to decrement fuel as for deep space

	; process asteroids
asters	set 7,(hl)	; set the "seen" bit
	call a_anim	; animation with asteroid noise
	call rand16	; get a random number 1..shfuel
	ld a,(randsd)	;
	ld d,a		;
	ld a,(shfuel)	; 
	dec a		; 
	jr z,z_fuel	; don't randomise fuel, as we've run out
	ld e,a		; 
	call div8	; 
	inc e		; E contains random number
	ld a,(shfuel)	; recall fuel level
	sub e		; subtract E from it
z_fuel	ld (shfuel),a	; store new fuel figures
	jr putshp	; done with asteroids

	; process planet
planet	set 7,(hl)	; set the "seen" bit
	call p_anim	; animation with planet noise
	ld a,$12	; take 18 units of fuel ...
	ld (shfuel),a	; ... and put them in the fuel tank
	jr putshp	; done with planet

	; process deep space
dspace	ld a,(beacxy)	; first check where the beacon is
	cp c		; is it here?
	jr nz,dsckfl	; no, on to check fuel level
	set 7,(hl)	; set the "seen" bit
	call s_anim	; flash with success noise
	jr dspafl	; on to set this square as seen
dsckfl	ld a,(shfuel)	; see how much fuel is left
	dec a		; is the ship about to run out?
	jr nz,dscksn	; no, on to check if the square is seen
	set 7,(hl)	; set the "seen" bit
	call f_anim	; flash with failure noise
	jr dspafl	; on to set this square as seen
dscksn	bit 7,(hl)	; has the player seen this space before?
	jr nz,dspafl	; yes, skip the animation
	set 7,(hl)	; set the "seen" bit
	call d_anim	; flash with deep space noise
dspafl	ld hl,shfuel	; point at fuel
	dec (hl)	; decrement it

	; put the ship down in its new location
putshp	ld a,c		; what is its new location?
	ld (rescxy),a	; put the ship there
	call drstat	; update the message and stats

	; check for success
	ld a,(beacxy)	; where is the beacon
	cp c		; is it in the same place as the ship?
	jr z,succss	; yes, so show success message
	call drship	; finally ensure the ship is visible again

	; now check for failure
	ld a,(shfuel)	; what is our fuel?
	or a		; do we have any left?
	jr nz,endmov	; yes, so just go on to end of routine

	; show failure message
failur	ld hl,faimsg	; point to failure message
	jr shoend	; show as end-of-level message

	; show success message
succss	call drship	; show the ship at its new location
	ld hl,(mssion)	; check to see what the current mission is
	;ld hl,999	; check the victory screen
	ld de,999	; we need to test it against 999
	or a		; clear carry flag
	sbc hl,de	; is level number 999?
	jp z,victry	; yes, the game is over
	ld hl,sucmsg	; point to success message
	jr shoend	; go on to print it

shoend	ld c,8		; six characters per line
	ld de,$4a6c	; point to lower half row 5 of map
	call msgprc	; print these six characters
	ld c,8		; six characters on second line
	ld de,$4b6c	; point to upper half row 6 of map
	call msgprc	; print second half of message

	ld hl,firmsg	; point to "press fire" message
	call msgprt	; print it in the normal place
	call ctlrel	; ensure no controls are activated
endfir	call ctlact	; now wait for fire to be pressed
	bit 4,a		;
	jr z,endfir	;
	call dr_map	; redraw the map

	; and that's it
endmov	pop hl		; retrieve entry values of registers
	pop de		;
	pop bc		;
	ret		; return to calling program

	;
	; Game calculations
	;

	; calculate the signal strength
	; outputs:
	;   A = strength of signal
signal	push bc		; store entry values of registers we corrupt here
	push de		;
	push hl		;
	ld a,(rescxy)	; see where the rescue ship is
	ld d,a		; convert this to row/column pair
	ld e,10		;
	call div8	;
	ex de,hl	; put these values aside
	ld a,(beacxy)	; see where the beacon is
	ld d,a		; convert this to row/column pair
	ld e,10		;
	call div8	;
	ld a,d		; find difference between rows
	sub h		;
	jr nc,rowpos	;
	cpl		;
	inc a		;
rowpos	ld b,a		; put row difference aside
	ld a,e		; find difference between columns
	sub l		;
	jr nc,colpos	;
	cpl		;
	inc a		;
colpos	ld c,a		; put column difference aside
	ld a,$12	; maximum signal is 18
	sub b		; subtract row distance
	sub c		; subtract column distance
	pop hl		; restore values of corrupted registers
	pop de		;
	pop bc		;
	ret		; done with calculation

	; convert map location to map data memory location
	; inputs:
	;   A = map location
	; outputs:
	;   HL = memory location
mapmem	push de		; we corrupt DE so store it
	ld e,a		; convert location A into an offset
	ld d,0		;
	ld hl,mapdat	; point HL to map data
	add hl,de	; add the offset
	pop de		; retrieve DE's entry value
	ret		; return to calling process

	;
	; Data for these routines
	;

	; messages for success and failure
sucmsg	.db "  SHIP   FOUND! "
faimsg	.db " OUT OF   FUEL  "
finmsg	.db "  GAME  FINISHED"
firmsg	.db " PRESS FIRE TO CONTINUE "

;	======================================================================
;
;	New Game and Level Ops
;
;	This part contains the routines for starting a new game, restarting
;	the current level and moving on to the next level, each of which
;	require confirmation in case of being selected accidentally. Also
;	in this section is the code to choose a level to complete.
;
	;
	; mission restart routine
	;

	; initialise ship positions
startm	ld a,(rscini)	; load accumulator from rescue ship position
	ld (rescxy),a	; store as current position
	ld a,18		; restore fuel to full level
	ld (shfuel),a	;
	call dr_map	; draw the map

	; finished
	jp maingm	; back to main game

	;
	; next mission routine
	;

	; check player has finished this mission
nextmn	ld a,(beacxy)	; where is the beacon?
	ld b,a		; store this information
	ld a,(rescxy)	; where is the rescue ship?
	cp b		; is it at the beacon?
	jr nz,incomp	; no, so this mission is incomplete

	; check player not finished last mission
	ld a,(mssion)	; lower byte of last mission ...
	cp $e7		; ... should be 231
	jr nz,admiss	; no, this is not the last mission
	ld a,(mssion+1)	; upper byte of last mission ...
	cp $03		; ... should be 3
	jr nz,admiss	; no, this is not the last mission
	ld hl,gmover	; point to "game over" message
	jr nexter	; go to print it
incomp	ld hl,incmsg	; point to "mission incomplete" message
nexter	call notprt	; display it as a notification
	jp maingm	; go back to main game

	; advance the mission
admiss	ld hl,(mssion)	; load the mission
	inc hl		; advance to the next mission
	ld a,(shfuel)	; what fuel have we left?
	ld e,a		;
	ld d,0		; we need this as a 16-bit value
	add hl,de	; add it to the mission number
	ld a,h		; approximately how far are we through the levels?
	cp $03		; are we near the end?
	jr nz,svmiss	; no, continue
	ld a,l		; exactly what mission have we advanced to?
	cp $e8		; have we shot past the last mission?
	jr c,svmiss	; no, continue
	ld hl,999	; yes, so we'll content ourselves with the last mission
svmiss	ld (mssion),hl	; save the mission number

	; all done
	jp newlvl	; back to new level routine before main game loop

	;
	; Mission Information Routine
	;

	; simply load HL with the mission number and call mission print sub
msinfo	ld hl,(mssion)	; recall the mission number
	call gncode	; generate mission code
	call mssprt	; print the information
	ld hl,($5c78)	; get random number from timer
	ld (randsd),hl	; save as next seed
	jp maingm	; back to game

	; display the mission number/code as a notification
	; inputs:
	;   HL = mission number
mssprt	push bc		; store soon-to-be-corrupted registers
	push de		;
	ld de,newmis+10	; point to digits of new mission message
	ld bc,100	; calculate the 100s
	call ldigit	; 
	ld bc,10	; calculate the 10s
	call ldigit	;
	ld a,l		; calculate the 1s
	add a,$30	;
	ld (de),a	;
	ld a,(lvcode)	; get level code
	ld l,a		;
	ld de,newmis+20	; point to pass-code of new mission message
	ld bc,10	; calculate the 10s
	call ldigit	;
	ld a,l		; calculate the 1s
	add a,$30	;
	ld (de),a	;
	ld hl,newmis	; point to new mission message
	call notprt	; print the notification
	pop de		; restore corrupted registers
	pop bc		;
	ret		; all done

	; calculate level digit
	; inputs:
	;   BC = digit to calculate (100, 10, 1)
	;   DE = address to save digit's ASCII value
	;   HL = number we're interested in - should not exceed 10*BC
	; outputs:
	;   A = ASCII value of digit
	;   DE = location to save next digit, if any
	;   (DE) = ASCII value of digit
ldigit	xor a		; initialise counter
tstdig	sbc hl,bc	; subtract column from value
	jr c,savdig	; save digit if we've gone negative
	inc a		; otherwise increase digit counter
	jr tstdig	; subtract again
savdig	add hl,bc	; pull our subject value back out of the negative
	add a,$30	; turn counter into an ASCII digit
	ld (de),a	; save this digit
	inc de		; prepare to save next digit, if called again
	ret		; all done - return

	;
	; Choose mission routine
	;

	; input mission number and code
chsemn	ld hl,newmis	; point to mission/code message
	call msgprt	; print it
	ld d,0		; zero upper byte of cursor offset
	ld e,10		; point cursor to character 10 of message
chloop
	call togglecr	; show (toggle) cursor
	;
ch_act	call ctlrel	; ensure no controls are activated
	call ctlact	; wait for a control to be activated
	bit 4,a		; is fire activated?
	jr nz,chdone	; yes, done
	bit 3,a		; is up activated?
	jr nz,ch__up	; yes, increase digit
	bit 2,a		; is down activated?
	jr nz,chdown	; yes, decrease digit
	bit 1,a		; is left activated?
	jr nz,chleft	; yes, move left

	call togglecr	; hide (toggle) cursor
	;
	inc e		; move cursor right
	ld a,13		; check the last level digit
	cp e		; have we moved past it?
	jr nz,ch_end	; no, check for end
	ld e,20		; yes, move to first digit of code
ch_end	ld a,22		; check last code digit
	cp e		; have we moved past it?
	jr nz,chloop	; no, back to loop
	ld e,10		; yes, back to first character
	jr chloop	; back to loop
chleft
	call togglecr	; hide (toggle) cursor
	;
	dec e		; move cursor left
	ld a,9		; check first level digit
	cp e		; have we moved past it?
	jr nz,chcode	; no, check first digit of code instead
	ld e,21		; yes, point to last digit of code
chcode	ld a,19		; check first code digit
	cp e		; have we moved past it?
	jr nz,chloop	; no, back to loop
	ld e,12		; yes, move to last digit of level
	jr chloop	; back to loop

	; decrease digit under cursor
chdown	ld hl,newmis	; point to new mission level/code message
	add hl,de	; advance to cursor position
	dec (hl)	; decrement the digit
	ld a,$2f	; check for the ASCII code before "0"
	cp (hl)		; have we reached it?
	jr nz,chupdm	; no, so on to update the message
	ld (hl),$39	; cycle the digit back to "9"
	jr chupdm	; on to update the message

	; increase digit under cursor
ch__up	ld hl,newmis	; point to the new mission level/code message
	add hl,de	; advance to cursor position
	inc (hl)	; increment the digit
	ld a,$3a	; check for ASCII code after "9"
	cp (hl)		; have we reached it?
	jr nz,chupdm	; no, go on to update the message
	ld (hl),$30	; cycle the digit back to 0

	; update the message after altering a digit
chupdm	ld hl,newmis	; point to the mission level/code message
	call msgprt	; print it
	jr chloop	; back to loop

	; done choosing - calculate mission and code numbers
chdone	;ld (hl),$44	; remove white cursor from message
	ld hl,0		; zero level calculation
	ld a,(newmis+10); check hundreds digit
	and $0f		; reduce to 0..9
	call ch_add	; multiply HL by 10 and add A
	ld a,(newmis+11); check tens digit
	and $0f		; reduce to 0..9
	call ch_add	; multiply HL by 10 and add A
	ld a,(newmis+12); check units digit
	and $0f		; reduce to 0..9
	call ch_add	; multiply HL by 10 and add A
	ex de,hl	; put mission number aside
	ld hl,0		; zero code calculation
	ld a,(newmis+20); check tens digit
	and $0f		; reduce to 0..9
	call ch_add	; append to code
	ld a,(newmis+21); check tens digit
	and $0f		; reduce to 0..9
	call ch_add	; append to code
	ex de,hl	; swap codes

	; check the level code
	call gncode	; generate the actual code for the proposed level
	cp e		; is it the same as the one entered?
	jr nz,ch_err	; no, so complain
	; go to desired mission
	ld (mssion),hl	; store new mission
	jp newlvl	; generate new level

	; erroneuos code entered
ch_err	ld hl,incode	; point to "invalid code" message
	call notprt	; print it
	ld hl,(mssion)	; check current mission
	call gncode	; regenerate its correct code (is this necessary?)
	jp maingm	; back to main game (temporary)

	; append a digit to the end of a number (like calculator input)
	; inputs:
	;    A - the digit to append
	;    HL - the value to append the digit to
	; outputs:
	;    HL - the updated value
ch_add	push de		; store DE before corrupting it
	add hl,hl	; multiply by 10
	push hl		;
	pop de		;
	add hl,hl	;
	add hl,hl	;
	add hl,de	;
	ld d,0		; transfer A to DE
	ld e,a		;
	add hl,de	; add digit to HL
	pop de		; restore uncorrupted DE
	ret		; all done

	;
	; Toggle the cursor
	;
togglecr
	push hl		; store the registers
	ld hl,msgpos	; point to message area on the screen
	add hl,de	; advance to area under cursor
	call chpatch	; replace the character
	pop hl		; restore the registers
	ret		;

	;
	; New Game Routine
	;

	; get confirmation and start new game
ngconf	ld hl,sgcnfm	; point to confirmation message
	call msgprt	; print it
	call ctlact	; wait for control to be activated
	bit 4,a		; was fire activated?
	jp nz,newgam	; yes, start new game
	jp maingm	; otherwise, continue game
	
	;
	; messages
	;

	; confirmation messages
slcnfm	.db "FIRE TO RESTART MISSION."
nlcnfm	.db " FIRE FOR NEXT MISSION. "
sgcnfm	.db "  FIRE TO RESTART GAME  "

	; notification messages
gmover	.db " ALL MISSIONS COMPLETED "
incmsg	.db "   MISSION INCOMPLETE   "
newmis	.db "  MISSION XXX: CODE XX  "
incode	.db "  INVALID MISSION CODE  "

;	======================================================================
;
;	Assembler code for the victory screen
;
	;
	; The victory screen routine
	;

	; draw the victory screen - pixels first
victry
	ld a,$A0		; there are 160 lines to draw
	ld b,$00		;
	ld de,HFILE+$0106	; point DE to screen address
	ld hl,vicscr		; point HL to victory screen graphics
vicrow
	ld c,$14		; there are 20 bytes per line
	ldir			; transfer those bytes to the screen
	;
	ex de,hl		;
	ld c,$0c		; offset to the next line
	add hl,bc		; on the screen
	ex de,hl		;
	;
	dec a			; continue to transfer
	jr nz,vicrow		; pixel rows until done

	; wait for FIRE to be pressed
	ld hl,sgcnfm	; point to "FIRE TO RESTART GAME" message
	call msgprt	; print it
	call ctlrel	; wait for all controls to be released
	call ctlfir	; wait for FIRE to be activated
	jp newgam	; new game

	;
	; The Graphics Data
	;

	; line based output of pixel data:
vicscr
	.db $00,$00,$00,$14,$00,$08,$00,$00,$00,$04,$15,$00,$23,$00,$00,$00,$00,$00,$00,$00
	.db $20,$00,$00,$16,$80,$10,$00,$00,$00,$02,$0A,$A0,$28,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$49,$00,$0C,$00,$00,$20,$08,$29,$00,$05,$80,$00,$00,$00,$00,$00,$00
	.db $00,$04,$00,$A4,$80,$38,$00,$00,$00,$00,$08,$4A,$05,$D0,$00,$10,$00,$04,$00,$20
	.db $00,$00,$02,$B9,$00,$14,$00,$00,$00,$40,$02,$00,$01,$80,$00,$00,$00,$08,$00,$50
	.db $80,$00,$12,$4A,$00,$3C,$00,$00,$00,$34,$00,$AA,$21,$A0,$00,$00,$00,$02,$00,$0A
	.db $00,$00,$0A,$01,$00,$1E,$20,$00,$00,$A4,$0A,$C0,$8B,$00,$00,$00,$00,$09,$01,$55
	.db $80,$00,$45,$00,$00,$5C,$00,$00,$00,$54,$05,$E8,$09,$80,$80,$00,$00,$01,$40,$A2
	.db $00,$00,$2A,$00,$00,$F7,$00,$00,$05,$24,$E5,$1D,$B9,$00,$20,$00,$00,$08,$07,$58
	.db $00,$00,$25,$00,$01,$F7,$80,$00,$04,$00,$F1,$8F,$A4,$02,$00,$00,$00,$00,$01,$C8
	.db $00,$00,$0A,$00,$37,$E7,$A8,$00,$0A,$20,$EA,$2A,$D2,$80,$00,$00,$00,$00,$02,$20
	.db $00,$05,$0B,$05,$DF,$03,$FC,$00,$01,$10,$58,$4A,$92,$40,$00,$00,$20,$00,$02,$94
	.db $00,$06,$94,$00,$3F,$C0,$FF,$00,$08,$22,$65,$90,$6A,$80,$00,$00,$00,$00,$02,$40
	.db $00,$03,$EE,$25,$0B,$E7,$E4,$00,$05,$10,$50,$54,$5D,$40,$00,$00,$00,$00,$00,$22
	.db $08,$05,$5A,$80,$01,$EF,$40,$00,$00,$52,$D2,$A8,$2A,$00,$00,$00,$00,$00,$0A,$00
	.db $00,$41,$6D,$70,$00,$EF,$00,$00,$02,$2A,$D0,$55,$28,$80,$00,$00,$00,$00,$41,$40
	.db $00,$01,$2A,$10,$00,$3C,$00,$04,$00,$5A,$45,$A8,$90,$00,$00,$00,$00,$00,$00,$00
	.db $01,$20,$04,$90,$00,$78,$00,$00,$00,$4D,$A5,$AC,$54,$00,$00,$00,$00,$00,$00,$A0
	.db $00,$08,$B1,$00,$00,$1C,$00,$00,$00,$2B,$10,$B2,$20,$00,$08,$00,$00,$00,$00,$08
	.db $00,$04,$08,$00,$00,$3C,$00,$00,$00,$25,$9A,$A9,$28,$00,$00,$00,$00,$00,$00,$00
	.db $00,$02,$24,$00,$00,$18,$00,$00,$00,$0A,$24,$A4,$80,$00,$00,$00,$04,$00,$00,$5A
	.db $80,$91,$0A,$80,$00,$30,$00,$00,$04,$02,$AA,$94,$08,$00,$00,$00,$00,$00,$00,$0A
	.db $00,$4A,$55,$00,$00,$08,$00,$20,$00,$4A,$44,$42,$00,$00,$00,$00,$00,$00,$00,$35
	.db $C4,$49,$0A,$80,$00,$10,$00,$00,$00,$05,$53,$94,$00,$00,$00,$00,$00,$00,$40,$0A
	.db $10,$29,$26,$80,$00,$00,$00,$00,$00,$56,$6A,$82,$00,$00,$00,$00,$00,$00,$10,$05
	.db $5C,$04,$32,$00,$00,$00,$00,$00,$00,$0B,$B4,$50,$00,$00,$00,$00,$00,$00,$00,$05
	.db $44,$29,$54,$00,$20,$00,$00,$00,$04,$A7,$D2,$00,$00,$00,$00,$00,$00,$00,$12,$85
	.db $2A,$4A,$50,$00,$00,$00,$00,$00,$04,$9B,$6A,$00,$00,$00,$00,$00,$00,$00,$04,$A5
	.db $AA,$25,$68,$20,$00,$00,$00,$00,$02,$6F,$A0,$00,$00,$20,$00,$00,$00,$00,$0B,$48
	.db $31,$10,$AA,$00,$00,$40,$00,$00,$02,$51,$D4,$00,$00,$00,$00,$00,$20,$00,$06,$6A
	.db $48,$49,$6A,$20,$00,$00,$00,$00,$09,$AA,$20,$00,$00,$00,$00,$00,$00,$00,$15,$91
	.db $60,$24,$B5,$10,$00,$00,$40,$40,$51,$20,$00,$05,$00,$00,$00,$00,$00,$00,$00,$A8
	.db $50,$12,$D4,$40,$00,$00,$00,$02,$00,$00,$00,$04,$00,$00,$00,$00,$00,$00,$03,$A5
	.db $A8,$04,$94,$10,$00,$00,$11,$40,$00,$00,$00,$05,$00,$00,$00,$00,$00,$0A,$40,$50
	.db $40,$00,$68,$A0,$00,$00,$24,$0A,$00,$00,$00,$00,$00,$00,$00,$00,$00,$14,$01,$49
	.db $50,$01,$14,$15,$00,$02,$2A,$C5,$00,$00,$00,$00,$00,$00,$00,$08,$00,$54,$14,$28
	.db $00,$15,$50,$54,$80,$04,$94,$54,$06,$FC,$00,$00,$00,$00,$00,$00,$00,$2B,$49,$08
	.db $00,$16,$4C,$06,$80,$25,$15,$54,$12,$0B,$80,$00,$02,$00,$00,$00,$01,$54,$A4,$54
	.db $00,$AA,$01,$41,$24,$95,$08,$A8,$20,$00,$80,$00,$00,$00,$00,$00,$02,$F5,$5D,$00
	.db $00,$1A,$80,$20,$14,$2D,$84,$A0,$20,$00,$00,$00,$00,$00,$00,$00,$02,$BF,$7F,$50
	.db $2A,$5C,$49,$15,$40,$92,$00,$00,$20,$00,$40,$00,$00,$00,$00,$00,$02,$FB,$FE,$40
	.db $25,$4A,$29,$50,$20,$0B,$40,$00,$18,$07,$80,$00,$00,$00,$00,$00,$05,$EF,$A8,$D4
	.db $12,$B0,$14,$A8,$02,$04,$00,$10,$05,$50,$00,$00,$00,$00,$00,$01,$2A,$F4,$42,$E2
	.db $00,$AA,$05,$68,$A1,$02,$00,$00,$08,$04,$00,$00,$00,$00,$00,$01,$AA,$FB,$20,$E9
	.db $50,$A8,$B3,$B4,$A4,$08,$80,$00,$00,$02,$00,$00,$00,$08,$00,$02,$AA,$A9,$01,$C0
	.db $00,$4A,$29,$B6,$55,$02,$40,$00,$08,$02,$00,$00,$00,$00,$00,$01,$51,$EA,$88,$F0
	.db $80,$29,$55,$69,$52,$85,$00,$00,$00,$02,$00,$00,$00,$00,$00,$00,$45,$C0,$15,$D8
	.db $40,$49,$09,$50,$55,$A1,$00,$00,$10,$01,$00,$00,$00,$00,$00,$00,$12,$C0,$22,$F8
	.db $80,$00,$14,$48,$49,$44,$00,$00,$00,$01,$00,$00,$00,$00,$00,$01,$55,$A2,$9A,$FA
	.db $40,$00,$91,$24,$09,$11,$00,$00,$10,$01,$00,$00,$00,$00,$00,$00,$7E,$D2,$28,$E0
	.db $14,$10,$2C,$92,$10,$24,$40,$00,$00,$03,$00,$00,$00,$00,$00,$0A,$95,$A9,$14,$E8
	.db $04,$09,$0A,$92,$00,$09,$00,$00,$10,$00,$80,$10,$00,$01,$00,$12,$EF,$AA,$16,$EA
	.db $58,$12,$AB,$A9,$05,$05,$28,$1C,$00,$01,$80,$00,$00,$02,$20,$0E,$AD,$A3,$34,$CC
	.db $2C,$10,$AA,$E4,$01,$22,$49,$5F,$C0,$00,$80,$00,$02,$01,$00,$25,$6F,$FA,$08,$EB
	.db $A8,$02,$93,$2A,$00,$09,$7F,$6B,$C0,$01,$40,$00,$00,$00,$20,$56,$BB,$AB,$24,$F5
	.db $4F,$A0,$41,$52,$00,$18,$FF,$2A,$C0,$00,$C0,$00,$00,$00,$80,$15,$5A,$F9,$08,$D5
	.db $D5,$97,$0A,$48,$A0,$06,$7F,$55,$40,$01,$01,$FC,$00,$00,$00,$2A,$AD,$C9,$20,$A5
	.db $D4,$95,$A1,$20,$14,$00,$FE,$02,$A0,$03,$07,$FF,$00,$00,$40,$14,$AB,$C8,$1A,$A5
	.db $D3,$45,$9D,$41,$80,$81,$0E,$00,$40,$0D,$1F,$FF,$C0,$00,$05,$56,$07,$86,$24,$D1
	.db $C9,$D1,$AA,$A8,$3E,$02,$03,$00,$20,$2D,$3F,$FF,$F0,$00,$00,$A9,$43,$52,$AA,$95
	.db $57,$44,$AA,$85,$01,$50,$2C,$00,$00,$6B,$7F,$FF,$FC,$00,$05,$12,$02,$CB,$98,$D8
	.db $81,$62,$6E,$A0,$02,$42,$00,$80,$00,$AA,$FF,$FF,$FF,$00,$00,$50,$82,$F1,$64,$6E
	.db $4B,$28,$2B,$40,$00,$A9,$00,$60,$21,$55,$FF,$FF,$FF,$C0,$00,$88,$05,$BE,$B0,$BB
	.db $40,$95,$9A,$A0,$00,$91,$54,$10,$02,$AB,$FF,$FF,$FF,$E0,$00,$20,$02,$7D,$44,$DF
	.db $11,$94,$4C,$40,$00,$0A,$28,$10,$0C,$A8,$FF,$FF,$FF,$F0,$00,$A0,$01,$FA,$FE,$0A
	.db $48,$9B,$47,$00,$00,$01,$7F,$A0,$00,$94,$FF,$FF,$FF,$F8,$00,$00,$00,$B9,$6A,$50
	.db $A5,$D4,$A4,$04,$00,$07,$FF,$80,$05,$24,$FF,$FF,$FF,$FC,$40,$80,$00,$54,$54,$0A
	.db $30,$BA,$25,$05,$00,$01,$3F,$40,$03,$20,$FF,$FF,$FF,$FE,$00,$00,$01,$58,$2A,$49
	.db $EA,$54,$80,$02,$50,$00,$5F,$28,$00,$C8,$FF,$FF,$FF,$FE,$00,$00,$02,$54,$55,$12
	.db $BA,$D6,$00,$04,$10,$00,$4F,$0A,$40,$18,$FF,$FF,$FF,$FF,$00,$00,$01,$5A,$25,$86
	.db $E9,$18,$00,$02,$44,$00,$15,$00,$00,$00,$7F,$FF,$FF,$FF,$00,$00,$05,$54,$99,$AB
	.db $7C,$CE,$80,$00,$14,$00,$00,$01,$F8,$08,$7F,$FF,$FF,$FF,$00,$00,$00,$94,$54,$2D
	.db $04,$B1,$20,$00,$7A,$02,$51,$06,$A8,$00,$1F,$FF,$FF,$FF,$00,$00,$10,$20,$16,$05
	.db $42,$94,$80,$00,$12,$E0,$A8,$86,$68,$08,$07,$FF,$FF,$FE,$00,$00,$04,$01,$16,$00
	.db $15,$50,$00,$00,$09,$02,$51,$4A,$A8,$01,$00,$7F,$FF,$F8,$40,$00,$12,$82,$4C,$12
	.db $0A,$4A,$80,$00,$01,$41,$29,$22,$A8,$08,$10,$00,$0F,$C0,$20,$00,$00,$90,$2A,$10
	.db $09,$20,$00,$10,$02,$00,$88,$0A,$A8,$00,$00,$80,$00,$00,$60,$00,$05,$04,$1A,$00
	.db $04,$14,$84,$04,$00,$80,$01,$20,$28,$00,$00,$07,$00,$00,$C0,$10,$00,$02,$AD,$00
	.db $00,$90,$10,$00,$00,$0A,$01,$00,$90,$08,$00,$00,$00,$F8,$00,$00,$00,$00,$96,$00
	.db $80,$10,$08,$01,$00,$05,$00,$30,$14,$00,$00,$00,$00,$00,$00,$00,$00,$04,$96,$00
	.db $00,$00,$00,$00,$00,$00,$01,$F0,$02,$08,$00,$00,$00,$00,$60,$00,$21,$AD,$4A,$40
	.db $40,$00,$00,$00,$00,$00,$0F,$F8,$00,$00,$00,$00,$00,$0F,$F8,$00,$10,$A3,$89,$64
	.db $80,$00,$00,$00,$00,$00,$7F,$CC,$00,$08,$00,$00,$01,$FF,$EE,$00,$0A,$49,$55,$00
	.db $A0,$20,$00,$00,$00,$03,$FF,$7E,$00,$00,$00,$00,$1F,$FF,$FB,$80,$04,$44,$55,$64
	.db $40,$00,$00,$00,$00,$1F,$FF,$FF,$80,$08,$00,$01,$EA,$FF,$F8,$E0,$02,$52,$A9,$01
	.db $50,$00,$00,$00,$00,$FF,$F7,$DF,$C0,$00,$00,$0D,$36,$AF,$E7,$F8,$02,$16,$55,$51
	.db $40,$00,$04,$00,$07,$FF,$DF,$F7,$E0,$08,$01,$52,$DB,$77,$9F,$DC,$09,$AB,$A0,$05
	.db $40,$00,$00,$00,$3F,$FF,$FF,$FD,$F8,$00,$12,$55,$55,$B5,$BF,$F7,$01,$AB,$91,$41
	.db $00,$00,$00,$01,$FF,$FF,$FF,$FF,$FE,$08,$01,$2A,$AA,$AE,$7F,$FB,$80,$00,$20,$04
	.db $00,$00,$00,$0F,$FF,$EF,$FF,$FF,$BF,$00,$00,$82,$AA,$AA,$AF,$FE,$E0,$00,$80,$00
	.db $00,$00,$00,$7F,$FD,$7F,$FF,$FF,$E8,$06,$00,$29,$55,$55,$77,$FF,$60,$00,$80,$20
	.db $00,$00,$03,$FF,$FD,$FF,$FF,$FF,$80,$00,$80,$00,$15,$55,$5A,$FF,$F8,$00,$40,$09
	.db $00,$00,$1F,$FF,$F7,$BF,$FF,$F8,$0A,$00,$20,$01,$4A,$A9,$6F,$7F,$EC,$00,$40,$10
	.db $20,$00,$BF,$FF,$DF,$DF,$FF,$80,$50,$00,$08,$00,$20,$A9,$55,$AF,$F7,$00,$41,$04
	.db $00,$01,$DF,$FE,$7F,$FF,$F8,$05,$00,$00,$03,$00,$0A,$45,$55,$77,$FB,$00,$00,$A0
	.db $00,$01,$FF,$FB,$FF,$FB,$80,$50,$00,$00,$00,$40,$00,$25,$55,$5A,$FD,$C0,$40,$68
	.db $04,$02,$FF,$EF,$FF,$F8,$05,$00,$00,$10,$00,$18,$00,$49,$55,$57,$7F,$A0,$41,$10
	.db $00,$01,$7F,$BF,$FF,$80,$50,$00,$00,$04,$00,$02,$00,$04,$55,$55,$AF,$F0,$80,$AA
	.db $00,$01,$5D,$FF,$F8,$05,$00,$00,$00,$02,$00,$00,$C0,$21,$15,$55,$77,$A0,$48,$10
	.db $00,$02,$BF,$FF,$80,$50,$00,$20,$00,$00,$80,$00,$18,$00,$41,$55,$5A,$D8,$24,$08
	.db $00,$02,$5E,$F8,$05,$00,$00,$00,$00,$00,$20,$00,$03,$20,$0A,$55,$57,$68,$01,$50
	.db $08,$01,$2F,$80,$50,$00,$00,$00,$08,$00,$08,$00,$00,$40,$01,$15,$55,$B4,$00,$20
	.db $00,$01,$37,$85,$00,$00,$00,$00,$00,$00,$02,$00,$00,$00,$00,$22,$AA,$D4,$01,$48
	.db $00,$00,$97,$90,$00,$00,$00,$00,$00,$00,$01,$00,$00,$00,$00,$08,$55,$5A,$04,$A4
	.db $00,$00,$8B,$E0,$00,$A1,$00,$00,$00,$00,$00,$50,$00,$00,$00,$00,$95,$54,$02,$88
	.db $00,$00,$45,$C0,$00,$18,$00,$00,$00,$00,$00,$04,$00,$00,$00,$00,$01,$54,$09,$6A
	.db $00,$00,$23,$E0,$00,$4A,$20,$00,$00,$00,$04,$0A,$80,$00,$00,$00,$02,$29,$24,$80
	.db $00,$00,$10,$60,$00,$05,$00,$00,$00,$00,$02,$01,$20,$00,$00,$00,$00,$00,$24,$34
	.db $80,$08,$08,$C0,$00,$12,$00,$00,$00,$00,$04,$00,$14,$00,$00,$00,$00,$01,$12,$24
	.db $00,$00,$06,$10,$00,$00,$80,$00,$00,$00,$00,$00,$02,$80,$00,$00,$00,$00,$25,$0A
	.db $80,$00,$01,$E0,$00,$00,$20,$00,$00,$00,$00,$00,$00,$50,$00,$00,$00,$02,$55,$52
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$02,$00,$2A,$00,$00,$00,$04,$5A,$89
	.db $40,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$28,$00,$0A,$A8,$00,$00,$10,$AA,$52
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$01,$04,$00,$00,$AD,$40,$01,$40,$C9,$49
	.db $80,$00,$00,$00,$00,$00,$00,$02,$00,$00,$00,$30,$01,$10,$12,$B5,$55,$03,$04,$92
	.db $00,$00,$00,$00,$16,$00,$00,$00,$00,$00,$00,$8A,$00,$04,$00,$4A,$A0,$05,$52,$02
	.db $80,$00,$00,$00,$08,$00,$00,$00,$00,$00,$00,$50,$00,$25,$00,$00,$00,$5E,$A8,$02
	.db $00,$00,$00,$00,$0A,$00,$00,$00,$00,$08,$00,$08,$00,$11,$2A,$00,$0A,$17,$00,$01
	.db $00,$20,$00,$00,$20,$00,$00,$20,$00,$00,$20,$20,$01,$54,$AA,$00,$51,$05,$00,$0A
	.db $00,$00,$00,$00,$20,$80,$00,$00,$00,$00,$10,$00,$01,$54,$82,$80,$44,$DE,$A0,$09
	.db $00,$00,$20,$00,$02,$00,$00,$00,$00,$00,$40,$02,$04,$95,$50,$02,$AA,$B5,$20,$54
	.db $00,$00,$10,$00,$22,$C0,$00,$00,$00,$00,$20,$01,$A0,$94,$40,$28,$4A,$FD,$A4,$22
	.db $00,$00,$00,$01,$0A,$80,$00,$00,$00,$00,$00,$05,$00,$4A,$95,$15,$2F,$2E,$41,$14
	.db $00,$00,$80,$08,$41,$40,$00,$00,$00,$00,$00,$00,$00,$22,$40,$89,$04,$B5,$A0,$80
	.db $00,$00,$40,$05,$54,$00,$00,$00,$00,$01,$10,$00,$20,$15,$20,$24,$B5,$12,$00,$00
	.db $88,$00,$14,$12,$52,$00,$00,$00,$04,$00,$00,$20,$08,$50,$80,$04,$94,$A8,$00,$00
	.db $00,$00,$02,$00,$A8,$00,$00,$00,$00,$00,$00,$28,$48,$25,$04,$12,$A5,$10,$01,$02
	.db $80,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $3C,$78,$91,$E3,$C7,$8F,$12,$20,$78,$F3,$CF,$12,$3C,$00,$F1,$E3,$C7,$89,$1E,$00
	.db $66,$CD,$DB,$36,$6C,$DF,$B3,$60,$CD,$FB,$D9,$BB,$60,$01,$9B,$16,$0C,$D9,$B1,$00
	.db $60,$CD,$FB,$06,$6C,$C6,$33,$60,$CC,$61,$99,$BF,$3C,$01,$9B,$C3,$CC,$19,$BC,$00
	.db $60,$CD,$BB,$67,$CF,$C6,$33,$60,$FC,$61,$99,$B7,$06,$01,$F3,$00,$6C,$19,$B0,$00
	.db $66,$CD,$9B,$36,$CC,$C6,$33,$7E,$CC,$63,$D9,$B3,$66,$C1,$B3,$16,$6C,$D9,$B1,$00
	.db $3C,$78,$91,$E2,$6C,$C6,$1E,$3C,$CC,$63,$CF,$12,$3C,$80,$99,$E3,$C7,$8F,$1E,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $3C,$F2,$07,$8F,$00,$24,$78,$90,$12,$3C,$48,$F0,$1E,$3C,$84,$F1,$03,$C7,$8F,$1C
	.db $66,$F6,$0C,$DF,$80,$66,$CD,$98,$33,$66,$CD,$88,$33,$66,$CD,$9B,$06,$2F,$D8,$B6
	.db $66,$66,$0C,$C6,$00,$7E,$CD,$98,$3F,$66,$CD,$E0,$30,$66,$FD,$9B,$07,$83,$1E,$33
	.db $7C,$66,$0C,$C6,$00,$3C,$CD,$98,$33,$7E,$CD,$80,$30,$66,$FD,$F3,$06,$03,$18,$33
	.db $60,$F7,$EC,$C6,$30,$18,$CD,$98,$33,$66,$79,$88,$33,$66,$CD,$83,$F6,$23,$18,$B6
	.db $20,$F3,$C7,$86,$30,$18,$78,$F0,$12,$66,$30,$F0,$1E,$3C,$48,$81,$E3,$C3,$0F,$1C
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $3C,$48,$F0,$12,$3C,$78,$E1,$E3,$C7,$81,$0B,$CF,$1E,$79,$E2,$47,$80,$1E,$24,$70
	.db $7E,$CD,$88,$33,$66,$CD,$B3,$16,$0F,$C1,$9B,$D8,$30,$7B,$37,$6C,$00,$33,$76,$D8
	.db $18,$FD,$E0,$3F,$66,$CD,$9B,$C3,$C3,$01,$F9,$8F,$1E,$33,$37,$E7,$80,$33,$7E,$CC
	.db $18,$CD,$80,$33,$7E,$F9,$9B,$00,$63,$01,$F9,$81,$83,$33,$36,$E0,$C0,$3F,$6E,$CC
	.db $18,$CD,$88,$33,$66,$D9,$B3,$16,$63,$01,$9B,$D9,$B3,$7B,$36,$6C,$D8,$33,$66,$D8
	.db $18,$48,$F0,$12,$66,$4C,$E1,$E3,$C3,$00,$93,$CF,$1E,$79,$E2,$47,$90,$33,$24,$70
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00
	.db $3C,$78,$F1,$23,$C7,$00,$91,$E2,$47,$80,$F1,$E3,$CF,$3C,$79,$09,$E2,$47,$80,$00
	.db $62,$CD,$9B,$B6,$2D,$81,$9B,$36,$6C,$C1,$9B,$17,$EF,$66,$C5,$9B,$17,$6F,$C0,$00
	.db $78,$CD,$9B,$F7,$8C,$C1,$FB,$36,$6C,$C1,$9B,$C1,$86,$66,$F1,$FB,$C7,$E3,$00,$00
	.db $60,$FD,$F3,$76,$0C,$C0,$F3,$36,$6F,$81,$F3,$01,$86,$7C,$C1,$FB,$06,$E3,$00,$00
	.db $62,$CD,$B3,$36,$2D,$80,$63,$36,$6D,$81,$B3,$11,$8F,$6C,$C5,$9B,$16,$63,$18,$00
	.db $3C,$CC,$99,$23,$C7,$00,$61,$E3,$C4,$C0,$99,$E1,$8F,$26,$78,$91,$E2,$43,$18,$00
	.db $00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00,$00

;	======================================================================
;
;	patch a character
;
chpatch
	push bc		; preserve the register values
	push de		; 
	;
	ld de,charmask	; patch address
	ld bc,$0820	; 8 pixel lines in a character (+ offset in C reg.)

	; fill a single pixel row at a time
chpatchl
	ld a,(de)	; load pixel row in character set
	xor (hl)	; change pixel row on screen, then
	ld (hl),a	; save onto screen
	ld a,b		; store the counter
	ld b,0		;
	add hl,bc	; point HL to next pixel row on screen
	ld b,a		; restore the counter
	inc de		; point DE to next pixel row in character set
	djnz chpatchl	; transfer remaining pixel rows
	;
	pop de		; restore the register values
	pop bc		;
	ret		; done
;
; ============================================================================
;
;	Output message text of fixed length
;
	; notification print - delay after the printing of the message
	;   HL = address of fixed-length message to print
	;   constant MSGPOS = screen address of first character top pixels
	;   constant MSGLEN = length of the message display line
notprt	push bc		; store the registers this routine overwrites
	call msgprt	; print the message
	ld b,100	; the delay will be 100/50ths second
notpr0
	call waitfrm	;
	;
	djnz notpr0	; continue till 100 interrupts have passed
	pop bc		; retrieve value of BC
	ret		; all done

	;
	; waits next frame (~1/50sec)
	; 
waitfrm
	push af		;
	push hl		;
	;
	ld hl,FRAMES	; LSB of counter (sv FRAMES)
	ld a,(hl)	; store
waitchg
	cp (hl)		; it is changed?
	jr z,waitchg	; no, test again
	;
	pop hl		;
	pop af		;
	ret		; yes -> return

	;
	; standard message filling entire message display line
	;   HL = address of fixed-length message to print
	;   constant MSGPOS = screen address of first character top pixels
	;   constant MSGLEN = length of the message display line
msgprt	push bc		; save BC before we overwrite it
	push de		; save DE before we overwrite it
	ld de,msgpos	; screen memory where messages will appear
	ld c,msglen	; length of message display line
	call msgprc	; print these characters
	pop de		; retrieve DE as it was on entry
	pop bc		; retrieve BC as it was on entry
	ret		; all done

	; display a message of specified length at specified location
	;   HL = address of fixed-length message to print
	;   DE = screen address of first character, top pixel line
	;   C  = length of message
msgprc	ld a,(hl)	; load character into A
	push hl		; store our string address while we borrow HL
	push de		; store screen address temporarily
	sub $20		; subtract value of space from A
	ld l,a		; transfer to HL
	ld h,0		;
	add hl,hl	; turn HL into an offset to CHRSET
	add hl,hl	;
	add hl,hl	;
	ld de,chrset	; point to character set
	add hl,de	; add the offset
	pop de		; regain our screen address
	ld b,$08	; 8 pixel lines in a character
	push de		; push our screen address again

	; fill a single pixel row at a time
msgprl	ld a,(hl)	; load character data
	ld (de),a	; save onto screen
	inc hl		; point HL to next pixel row in character set
	push hl		; store our pixel pointer
	ld hl,$0020	; the offset
	add hl,de	;
	ex de,hl	; point DE to next pixel row on screen
	pop hl		; retrieve pixel pointer
	djnz msgprl	; transfer remaining pixel rows
	pop de		; restore DE to the correct row in screen memory
	inc e		; point DE to next character in screen memory
	pop hl		; retrieve character pointer
	inc hl		; increment it to point to the next character
	dec c		; set character count
	jr nz,msgprc	; print remaining characters
	ret		; all done, return
;
;	======================================================================
;
;	Generic flashing animation routine
;
	; simple flash animation alternating between ship and map feature
	; inputs:
	;   HL = address of delay/sound routine to use
flanim	push bc		; store entry values of registers
	ld b,2		; initialise the flash counter
flloop	call drship	; draw the ship
	call flwait	; wait for 1/4 second with appropriate noise
	ld a,(rescxy)	; point to ship location
	call drtile	; draw the tile behind the ship
	call flwait	; wait for 1/4 second with appropriate noise
	djnz flloop	; repeat animation until done
	pop bc		; retrieve values of registers
	pop hl		; (HL was pushed by calling process)
	ret		; all done

	; dummy subroutine, transfer straight to (HL)
flwait	jp (hl)		; code at destination will have a RET at the end

	;
	; Deep Space Animations
	;

	; main animation routine
d_anim	push hl		; store HL before overwriting
	ld hl,d_wait	; assign silent wait routine
	jp flanim	; standard flash animation

	; delay routine for deep space (i.e. silence)
d_wait	push bc		; store values of registers before overwriting
	ld b,$0c	; approximate 1/4 second, or 1/5 for 60Hz systems
d_wtlp
	call waitfrm	;
	;
	djnz d_wtlp	; count down until delay done
	pop bc		; retrieve overwritten values
	ret		; return to calling routine

	; animation routine on successful mission
s_anim	push hl		; store HL before overwriting
	ld hl,$800c	; low-pitched beep
	ld (s_wksp),hl	; store this
	ld hl,s_wait	; assign success noise wait routine
	jp flanim	; standard flash animation

	; delay routine for successful mission
s_wait	push bc		; store values of registers before overwriting
	push hl		;
	ld b,$0c	; approximate 1/4 second, or 1/5 for 60Hz systems
	ld hl,(s_wksp)	; load pitch
s_wtlp
	call waitfrm	;
	;
	call tone	; make a noise
	dec h		; raise the tone
	djnz s_wtlp	; count down until delay done
	ld (s_wksp),hl	; store this
	pop hl		; retrieve register values
	pop bc		;
	ret		; return to calling routine

	; workspace
s_wksp	.db 0,0		; pitch and length of beep

	; animation routine on failed mission
f_anim	push hl		; store HL before overwriting
	ld hl,$c00c	; high-pitched beep
	ld (f_wksp),hl	; store this
	ld hl,f_wait	; assign success noise wait routine
	jp flanim	; standard flash animation

	; delay routine for failed mission
f_wait	push bc		; store values of registers before overwriting
	push hl		;
	ld b,$0c	; approximate 1/4 second, or 1/5 for 60Hz systems
	ld hl,(f_wksp)	; load pitch
f_wtlp
	call waitfrm	;
	;
	call tone	; make a noise
	inc h		; lower the tone (!)
	djnz f_wtlp	; count down until delay done
	ld (f_wksp),hl	; store this
	pop hl		; retrieve register values
	pop bc		;
	ret		; return to calling routine

	; workspace
f_wksp	.db 0,0		; pitch and length of beep

	;
	; Planet animation
	;

	; planet animation routine
p_anim	push hl		; store HL before overwriting
	ld hl,p_wait	; use planet sound/wait routine
	jp flanim	; on to standard flash animation

	; delay with noise for planet animation
p_wait	push bc		; store values of registers before overwriting
	push hl		;
	ld b,$0c	; approximate 1/4 second, or 1/5 for 60Hz systems
	ld hl,$ff08	; low-pitched rumbling noise
	call tone	; make a noise
p_wtlp
	call waitfrm	;
	;
	djnz p_wtlp	; count down until delay done
	pop hl		; retrieve register values
	pop bc		;
	ret		; return to calling routine

	;
	; Asteroid animation
	;

	; main animation routine
a_anim	push hl		; store HL before overwriting
	ld hl,a_wait	; use asteroid sound/wait routine
	jp flanim	; on to standard flash animation

	; delay routine with noise for asteroid animation
a_wait	push bc		; store values of registers before overwriting
	push hl		;
	ld b,$0c	; approximate 1/4 second, or 1/5 for 60Hz systems
	ld hl,$ff08	; low-pitched rumbling noise
a_wtlp
	call waitfrm	;
	;
	bit 2,b		; should we have silence?
	jr nz,a_slnt	; yes
	call noise	; make a noise
a_slnt	djnz a_wtlp	; count down until delay done
	pop hl		; retrieve register values
	pop bc		;
	ret		; return to calling routine

	;
	; Hyperspace animations
	;

	; initial spinning animation
h1anim	push bc		; store registers about to be corrupted
	push de		;
	push hl		;
	ld d,0		; zero upper half of offset
	ld b,8		; we rotate the ship 8 times
	ld a,(rescdr)	; what direction are we facing?
	rrca		; change bits 0/1 to a value 0..3 to correspond ...
	rrca		; ... with sprite numbers right,up,down,left
	ld hl,$0108	; point HL to start of noise generator data
h1loop	ld e,a		; cycle through values 00->10->11->01->00 ...
	rr e		; ... or right->down->left->up->right
	rla		;
	rr e		;
	rla		;
	xor $02		;
	and $03		;
	ld c,a		; preserve temporary direction
	call drrotn	; show the ship rotated
	ld a,c		; restore temporary direction
	ld c,b		; save rotation counter for a moment
	ld b,6 		; delay approximately 1/8 second, 6 frames
h1wait
	call waitfrm	;
	;
	call noise	; make a noise
	inc h		; lower the pitch
	inc h		;
	inc h		;
	inc h		;
	djnz h1wait	; continue waiting until delay done
	ld b,c		; retrieve rotation counter
	djnz h1loop	; continue rotating until done
	pop hl		; retrieve corrupted registers
	pop de		;
	pop bc		;
	ret		; all done

	; secondary animation after movement
h2anim	push hl		; store HL before overwriting
	ld hl,h2wait	; use hyperspace sound/wait routine
	jp flanim	; on to standard flash animation

	; draw rotated rescue ship
drrotn	push de		; store on the stack the same registers ...
	push hl		; ... as does DRTILE below
	push af		; store temporary rotated ship tile number
	ld a,(rescxy)	; where is the ship?
	ld e,a		; store this location for later use
	ld d,0		; zero upper half of location word
	pop af		; recall rotated ship tile number
	jp drskip	; from now on act as DRTILE

	; delay and noise for secondary hyperspace animation
h2wait	push bc		; store values of registers before overwriting
	ld b,$0c	; approximate 1/4 second, or 1/5 for 60Hz systems
h2wtlp
	call waitfrm	;
	;
	djnz h2wtlp	; count down until delay done
	pop bc		; retrieve overwritten values
	ret		; return to calling routine
;
;	======================================================================
;
;	Ship drawing routines
;
	;
	; Map drawing routines
	;

	; draw the whole map
dr_map
	push bc		; remember value of BC on entry
	ld b,100	; there are 100 locations ...
drloop	ld a,100	; calculate location
	sub b		;
	call drtile	; draw graphic tile at this location
	djnz drloop	; on to draw next location till done
	call drship	; draw the rescue ship
	pop bc		; recall value of BC on entry

	ret		; finished drawing

	; draw correct map graphic at location A
	; inputs:
	;   A = location to draw
drtile	push de		; store entry values of registers
	push hl		;
	ld e,a		; convert location into 16-bit offset
	ld d,0		; 
	ld hl,mapdat	; point HL at the map
	add hl,de	; add offset
	ld a,(hl)	; what's on the map?
	bit 7,a		; has it been seen?
	jr z,drblnk	; no, so it's blank
	add a,$04	; map features are 4..7
	and $7f		; reset the top bit
	jr drskip	; skip the code for blanking out the square
drblnk
	ld a,$09	; select the blank sprite
drskip
	ld (wktile),a	; temporarily store sprite number
	ld a,e		; recall map location
	push de		; store it on the stack - we need it twice
	call fupper	; get memory location for upper half of sprite
	ld a,(wktile)	; recall sprite number ...
	call drhalf	; ... and draw the top half
	pop de		; what was the location again?
	ld a,e		; transfer to A
	call flower	; get memory location for lower half of sprite
	ld a,(wktile)	; recall sprite number again
	or $80		; set to bit to signify we want to ...
	call drhalf	; ... draw the lower half
	pop hl		; retrieve entry values of registers
	pop de		;
	ret		; done with dawing tile

	; workspace
wktile	.db 0		; workspace for map tile

	; draw rescue ship graphic at appropriate location
drship	push de		; store on the stack the same registers ...
	push hl		; ... as does DRTILE below
	ld a,(rescxy)	; where is the ship?
	ld e,a		; store this location for later use
	ld d,0		; zero upper half of location word
	ld hl,beacxy	; point at the beacon location
	cp (hl)		; are they at the same place?
	jr z,drbeac	; yes, so we're drawing a ship/beacon
	ld a,(rescdr)	; otherwise see which way we're facing
	rrca		; change this to a value 0..3 to correspond ...
	rrca		; ... with sprite numbers right,up,down,left
	and $03		;
	jr drskip	; from now on act as DRTILE
drbeac	ld a,$08	; this is the sprite for two ships together
	jr drskip	; from now on act as DRTILE

	; find screen address for a given map location
	; inputs:
	;   A = map location
	; outputs:
	;   DE = screen memory location
	; entry points:
	;   FUPPER = entry to find upper half of location
	;   FLOWER = entry to find lower half of location
fupper	push hl		; store entry value of HL
	ld hl,upptbl	; point HL to table of upper row locations
	jr fdhalf	; the rest of the location is common to maplwr

	; entry point for lower half of map sqaure
flower	push hl		; store hl
	ld hl,lowtbl	; point HL to table of lower row locations
fdhalf
	push bc		; store entry value of BC
	ld d,a		; transfer map location
	ld e,10		; to be divided by row width
	call div8	; now D=row, E=column
			; A == E !!
	ld e,0		; DE:=2 x 8 x 32 x rows = 512x rows
	rl d		;
	add hl,de	; apply offset to point at screen row address
	
	rla		;
	ld d,e		; retrieve column number
	ld e,a		;
	add hl,de	; add it to the screen row address
	ex de,hl	; transfer result back into DE
	pop bc		; retrieve original values of registers
	pop hl		;
	ret		; done

	; screen address for 1st map row (upper)
upptbl	= HFILE + $0006
	; screen address for 1st map row (lower)
lowtbl	= upptbl + $0100

	; draw half a map graphic
	; inputs:
	;   A = hsssssss; h=top(0)/bottom(1); s=sprite number
	;   DE = screen address
drhalf	push bc		; save registers to be overwritten
	push hl		;
	sla a		; shift top bit of A into carry
	rl c		; and store in bottom bit of c
	srl a		; restore A without its top bit
	ld b,a		; store a into counter
	ld hl,mapgfx	; point to start of map sprite graphics
	push de		; remember screen address DE for a moment
	bit 0,c		; which half of the sprite are we printing?
	jr z,findsp	; find the right sprite
	ld de,$0010	; bottom, so skip top half of sprite data ...
	add hl,de	; ... by adding it to HL	
findsp	ld de,$0020	;$0024	; set D to length of sprite information (de)
	and b		; is counter 0?
	jr z,sfound	; if so, we're already pointing at the right sprite
nextsp	add hl,de	; otherwise point to next sprite ...
	djnz nextsp	; ... and check again

	; draw the pixels
sfound
	pop de		; retrieve screen address
	ex de,hl	;
	ld bc,$081F	; 8 lines for this half of sprite (offset in C reg.)
pixrow
	ld a,(de)	; transfer first byte from sprite to screen
	ld (hl),a	;
	inc de		;
	inc hl		;
	ld a,(de)	; transfer second byte from sprite to screen 
	ld (hl),a	;
	inc de		;
	ld a,b		; store the counter
	ld b,0		;
	add hl,bc	; point HL to next screen row
	ld b,a		; restore the counter
	djnz pixrow	; output the next row

	pop hl		; restore registers
	pop bc		;
	ret		; done with this half square

	;
	; Status drawing routines
	;

	; draw current map location and statistics
drstat	push de		; store entry values of registers
	push hl		; 
	;
	; draw the fuel gauge and indicator
	ld a,(shfuel)		; check current fuel again
	ld hl,HFILE+$0202	; point to fuel gauge area in pixels
	call drgaug		; draw the gauge
	;
	; draw the signal gauge and indicator
	call signal		; calculate the signal strength
	ld hl,HFILE+$021C	; point to signal gauge area in pixels
	call drgaug		; draw the gauge
	;
	; check for win or loss
statvc	ld a,(rescxy)	; where is the rescue ship?
	ld d,a		; remember this location
	ld a,(beacxy)	; where is the beacon?
	cp d		; is it in the same place?
	jr nz,statfl	; no, check for no fuel instead
	ld hl,bcnmsg	; otherwise point to "mission complete" message ...
	jr stamsg	; ... and go on to print it
statfl	ld a,(shfuel)	; what is our fuel level again?
	or a		; are we out of fuel?
	jr nz,drname	; no, print map feature name instead
	ld hl,nofmsg	; otherwise point to "out of fuel"
	jr stamsg	; go to print message

	; prepare status text with feature name
drname	ld a,(rescxy)	; where is the ship?
	ld e,a		; convert this to a 16-bit offset
	ld d,0		;
	ld hl,mapdat	; point to the map area
	add hl,de	; advance to the rescue ship's location
	ld a,(hl)	; what is there?
	add a,a		; multiply by 24 and convert to a 16-bit offset
	add a,a		;
	add a,a		;
	ld e,a		;
	add a,a		;
	add a,e		;
	ld e,a		;
	ld hl,ftrnam	; point to feature name table
	add hl,de	; advance to the appropriate feature

	; output the status text
stamsg	call msgprt	; print its name
	pop hl		; retrieve entry values of registers
	pop de		;
	ret		; all done

	; draw a gauge
	; inputs:
	;   A = value/level of gauge
	;   HL = address of top left attribute of gauge
drgaug
	push bc			; store entry values of registers
	push de			;
	ld (ggtopadr+1),hl	; $40a2 - fuel,	$40bc - signal
	;
	ld b,$12		; 18 is the maximum gauge value
	ld c,a			; store gauge value
ggerow
	ld a,$12		;
	sub b			; 
ggtopadr
	ld hl,$0000		; address of the top left byte of the gauge
	add a,h			; add the offset
	ld h,a			;
	ex de,hl		; screen address to DE
	;
	ld hl,ggblack		; gauge above its level is black
	ld a,c			; restore gauge value
	cp b			; is the gauge value this high?
	jr c,ggsame		; no, so we don't change colour yet
	inc hl			; otherwise the gauge is yellow from here on
	inc hl			;
ggsame
	push bc			;
	ld c,2			; paint the gauge,
	call msgprc		; left & right character
	;
	pop bc			;
	djnz ggerow		; paint the next row, if appropriate
	;
	pop de			; restore entry values of registers
	pop bc			;
	ret			;
ggblack
	.db $20,$20,$60,$61

	;
	; Graphics and text for drawing
	;

	; rescue ship moving right
mapgfx	.db $00,$00,$7E,$00,$38,$00,$2C,$00,$1E,$00,$5F,$80,$7E,$40,$7E,$20
	.db $5F,$F8,$7F,$F4,$70,$02,$5F,$FC,$1F,$80,$3F,$00,$7F,$80,$00,$00

	; rescue ship moving up
	.db $00,$00,$01,$80,$03,$C0,$02,$40,$06,$60,$04,$20,$0D,$B0,$56,$6A
	.db $75,$AE,$6A,$56,$6A,$56,$7D,$BE,$6E,$76,$42,$42,$01,$80,$00,$00

	; rescue ship moving down
	.db $00,$00,$01,$80,$43,$C2,$6A,$56,$7D,$BE,$6B,$D6,$6F,$F6,$74,$2E
	.db $54,$2A,$0E,$70,$06,$60,$07,$E0,$03,$C0,$03,$C0,$01,$80,$00,$00

	; rescue ship moving left
	.db $00,$00,$00,$7E,$00,$1C,$00,$34,$00,$78,$01,$EA,$02,$6E,$04,$6E
	.db $1F,$EA,$3F,$EE,$7E,$0E,$3F,$FA,$01,$F8,$00,$FC,$01,$FE,$00,$00

	; deep space
	.db $00,$00,$00,$08,$00,$00,$08,$00,$1C,$02,$08,$20,$00,$00,$00,$00
	.db $00,$00,$00,$02,$01,$00,$40,$00,$00,$00,$00,$08,$10,$00,$00,$00

	; planet
	.db $07,$E0,$1E,$F8,$3F,$A8,$7D,$F4,$7F,$5A,$FF,$EA,$EE,$A9,$FF,$EC
	.db $D5,$52,$FF,$DA,$AA,$51,$6B,$54,$5A,$A8,$0A,$A0,$14,$88,$02,$40

	; asteroids
	.db $00,$00,$06,$06,$36,$02,$30,$30,$00,$78,$48,$7A,$62,$30,$00,$00
	.db $00,$00,$62,$60,$70,$64,$30,$00,$06,$06,$06,$0E,$00,$0C,$00,$00

	; hyperspace
	.db $00,$00,$07,$60,$08,$10,$01,$00,$2A,$64,$48,$F2,$52,$F2,$12,$60
	.db $55,$0A,$42,$D2,$48,$02,$24,$20,$13,$88,$00,$10,$07,$60,$00,$00

	; rescue ship and stricken ship together
	.db $00,$00,$1C,$00,$3E,$00,$7F,$00,$7D,$00,$79,$00,$33,$00,$1F,$80
	.db $01,$FC,$60,$BA,$38,$D8,$34,$F8,$3E,$F0,$26,$80,$78,$40,$00,$00

	; blank map graphic (may be replaced with "unexplored" indicator)
	.db $00,$00,$00,$00,$3F,$FC,$20,$04,$2A,$A4,$25,$54,$2a,$a4,$25,$54
	.db $2A,$A4,$25,$54,$2A,$A4,$25,$54,$20,$04,$3F,$FC,$00,$00,$00,$00

	; names of map features
ftrnam	.db " YOU ARE IN DEEP SPACE  "
	.db "REFUELLED AT THE PLANET "
	.db "NAVIGATED THE ASTEROIDS "
	.db "WENT THROUGH HYPERSPACE "

	; end of mission messages
nofmsg	.db "  YOU ARE OUT OF FUEL!  "
bcnmsg	.db "    MISSION COMPLETE    "

;	======================================================================
;
;	If these routines are to be called for more than 1/50 second, then
;	interrupts should be disabled first.  NOISE and TONE may be called
;	for brief periods between interrupts.
;
	; pseudo-random noise
	; inputs:
	;   H = pitch, or delay between clicks
	;   L = length of sound, in clicks
noise	push af		; store registers about to be corrupted
	push bc		;
	push de		;
	ld c,l		; initialise counter for sound length
	ld de,(noisea)	; pick up noise data from where we left off
noislp	ld a,(de)	; load byte for noise
	;
	or $fe	  ; on/off chanel A
	out ($0f),a	;
	;
	ld b,h		; initialise the delay between clicks
noisdl	djnz noisdl	; count down
	inc de		; point to next data
	dec c		; decrement counter for sound length
	jr nz,noislp	; continue making sound till done
	ld a,d		; check high bit of noise address
	and $1f		; ensure it stays within the ROM
	ld d,a		;
	ld (noisea),de	; store current noise address for next time
	pop de		; retrieve uncorrupted register values
	pop bc		;
	pop af		;
	ret		; all done

	; data
noisea	.dw $0000	; address of random data to be used as noise

	; constant tone
	; inputs:
	;   H = pitch, or delay between clicks
	;   L = length of sound, in clicks
tone	push af		; store registers about to be corrupted
	push bc		;
	ld c,l		; initialise counter for sound length
	;
	ld a,$ff	;
tonelp	out ($0f),a	; on/off chanel A
	xor $01		;
	;
	ld b,h		; initialise the delay between clicks
tonedl	djnz tonedl	; count down
	dec c		; decrement counter for sound length
	jr nz,tonelp	; continue making sound till done
	pop bc		;
	pop af		;
	ret		; all done

	; ping
	; inputs:
	;    H = pitch, or delay between clicks
	;    L = length of sound, in clicks
ping
	call waitfrm	;
	;
	call tone	;
	dec l		;
	jr nz,ping	;
	ret		;

;	======================================================================
;
;	These control player input by keyboard or joystick.
;
;	Both routines store the status of the controls in a variable CTLOPT,
;	with the following bits set: bit 4 for FIRE, 3 for UP, 2 for DOWN,
;	1 for LEFT and 0 for RIGHT (values 16, 8, 4, 2, 1 respectively).
;	The POLl routines also leave this value in the accumulator.
;
	;
	; Wait routines
	;

	; wait until all controls are released
ctlrel	call ctlpol	; poll the controls
	or a		; is anything activated?
	jr nz,ctlrel	; yes, so continue waiting
	ret		; return when done

	; wait until any control is activated
ctlact	call ctlpol	; poll the controls
	or a		; is anything activated?
	jr z,ctlact	; no, so continue waiting
	ret		; return when done

	; wait until FIRE is activated
ctlfir	call ctlpol	; poll the controls
	bit 4,a		; is fire pressed?
	jr z,ctlfir	; no, go back to poll controls again
	ret		; otherwise return

	;
	; Poll routines
	;

	; entry address of poll routines
ctlpol			; usual address when called from elsewhere
	jr joypol	; joystick by default (+12Ts)
	;
	; poll the keyboard
	;   constant CTLOPT = address to save 5-bit activated controls to
	;   returns A = copy of byte saved to CTLOPT
kbdpol	push bc		; save BC which we overwrite
	push hl		; save HL which we overwrite
	ld b,5		; number of controls to scan
	ld hl,kbdtbl	; look at table of keystable of keys
kbdchk	ld a,(hl)	; which keyboard row to look at?
	inc hl		; then the next next byte ...
	in a,($fe)	; look at that keyboard row
	and (hl)	; ... isolates the specific key (Z flag = pressed)
	inc hl		; point at next byte without hurting flags
	ld a,(ctlopt)	; load the current value from CTLOPT
	jr nz,kbdrol	; key is up, leave carry reset
	scf		; key is down, set carry flag
kbdrol	rl a		; rotate carry flag into CTLOPT
	ld (ctlopt),a	; store value in CTLOPT
	djnz kbdchk	; check next key if appropriate
	pop hl		; retrieve HL as it was on entry
	pop bc		; retrieve BC as it was on entry
	ret		; return

kbdtbl	.db $fd,$01	; FIRE (A)
	.db $fd,$02	; UP (S)
	.db $fe,$04	; DOWN (X)
	.db $7f,$08	; LEFT (N)
	.db $7f,$04	; RIGHT (M)

	; poll the joystick
	;   constant CTLOPT = address to save 5-bit activated controls to
	;   returns A = copy of byte saved to CTLOPT

	;; KEMPSTON:	000FUDLR
joypol
	in a,($1f)	; check KEMPSTON inreface
	cp $ff		; found?
	jr nz,joypol2	; yes, store bits
	;
	;; ZXPAND:	UDLRF???
	;
	ld bc,$e007	; read the previously latched joystick value
	in a,(c)

	or $7		; bottom 3 bits are undefined (not strictly true ;)
	cpl		; ZXpand joy bits are active low, kempston are active high
	rrca		; CY=0
	rrca		; CY=0
	rrca		; CY=0
	rra		; CY=F
	jr nc,joypol1	;

	set 4,a		;
joypol1
	push af		;
	ld a,$a0	; start the joystick value latching process for next time
	out (c),a	;
	pop af		;
joypol2
	ld (ctlopt),a	; save bits directly to ctlopt
	ret

;	======================================================================
;
;	These routines generate a 16-bit random number, and perform 8-bit
;	modulus arithmetic to reduce the number to a useful range for the game.
;	Both routines have been adapted from code in the public domain.
;
	; 16-bit Random number generator slightly modified from PD
	;   constant RANDSD = address of seed and generated random number
rand16	push de		; store DE on stack
	push hl		; store HL on stack
	ld de,(randsd)	; seed should have been set from timer or some such
	ld a,d		; load A from seed high
	ld h,e		; load H from seed low
	ld l,$fd	; load L with 253
	or a		; reset carry flag
	sbc hl,de	; subtract seed from HL
	sbc a,0		; subtract carry from A
	sbc hl,de	; subtract seed from HL again
	ld d,0		; clear D
	sbc a,d		; subtract carry from A again
	ld e,a		; load D with A
	sbc hl,de	; subtract (D)E from HL
	jr nc,rand	; if HL positive, set random number now
	inc hl		; otherwise increase HL first
rand	ld (randsd),hl	; save over seed
	pop hl		; retrieve HL
	pop de		; retrieve DE
	ret		; return

	; Simple modulus code to ensure number in desired range
	;   A = 8-bit divisor
	;   constant RANDSD = address of seed and last generated number
randmo	push bc		; store BC on the stack
	push hl		; store HL on the stack
	ld hl,(randsd)	; retrieve random number
	res 7,h		; ensure positive
	ld c,a		; set BC to dividend
	ld b,0		; we're never interested in numbers > 255
randml	or a		; reset carry flag
	sbc hl,bc	; subtract dividend from HL
	jp p,randml	; keep doing it till negative
	add hl,bc	; add the divided back again
	ld a,l		; result back in A
	pop hl		; retrieve HL
	pop bc		; retrieve DE
	ret		; return
;
;	======================================================================
;
;	Integer 8-bit division
;
	; inputs:
	;   D = divisor
	;   E = dividend
	; outputs:
	;   D = quotient
	;   E = remainder
div8	ld a,d		; move divisor to accumulator
	ld d,0		; initialise quotient
div8lp	sub e		; loop - subtract dividend from divisor
	jr c,div8dn	; quit if we're below zero
	inc d		; otherwise increment quotient ...
	jr div8lp	; ... and go to subtract again
div8dn	add a,e		; restore remainder
	ld e,a		; store remainder in E
	ret		; all done
;
;	======================================================================
;
;	Integer 16-bit division
;
	; inputs:
	;   HL = divisor
	;   DE = dividend
	; outputs:
	;   HL = remainder
	;   DE = quotient
div16	push bc		; store BC as we'll be using it as the quotient
	ld bc,0		; initialise the quotient
	or a		; reset the carry flag
div16r	sbc hl,de	; loop - subtract dividend from divisor
	jr c,div16d	; quit if we're below zero
	inc bc		; otherwise increment quotient ...
	jr div16r	; ... and go to subtract again
div16d	add hl,de	; restore remainder
	ld d,b		; move quotient to DE
	ld e,c		;
	pop bc		; restore BC
	ret		; all done

;	======================================================================
chrset
	.db $00,$00,$00,$00,$00,$00,$00,$00 ; space
	.db $00,$18,$18,$18,$18,$00,$18,$00 ; exclamation mark
	.db $00,$66,$66,$00,$00,$00,$00,$00 ; quotes
	.db $00,$24,$7e,$66,$66,$7e,$24,$00 ; hash
	.db $00,$18,$3e,$70,$0e,$7c,$18,$00 ; dollar
	.db $00,$66,$6c,$18,$18,$36,$66,$00 ; percent
	.db $00,$38,$6c,$38,$6e,$6c,$3a,$00 ; ampersand
	.db $00,$18,$10,$00,$00,$00,$00,$00 ; apostrophe
	.db $00,$1c,$30,$30,$30,$30,$1c,$00 ; left bracket
	.db $00,$38,$0c,$0c,$0c,$0c,$38,$00 ; right bracket
	.db $00,$00,$66,$3c,$7e,$3c,$66,$00 ; asterisk
	.db $00,$00,$18,$18,$7e,$18,$18,$00 ; plus
	.db $00,$00,$00,$00,$00,$18,$10,$00 ; comma
	.db $00,$00,$00,$00,$3c,$00,$00,$00 ; hyphen
	.db $00,$00,$00,$00,$00,$18,$18,$00 ; full-stop
	.db $00,$06,$0c,$18,$18,$30,$60,$00 ; slash
	.db $00,$3c,$66,$6e,$76,$66,$3c,$00 ; 0
	.db $00,$38,$38,$18,$18,$3c,$3c,$00 ; 1
	.db $00,$3c,$66,$06,$3c,$60,$7e,$00 ; 2
	.db $00,$3c,$66,$0c,$06,$66,$3c,$00 ; 3
	.db $00,$0c,$1c,$2c,$6c,$7e,$0c,$00 ; 4
	.db $00,$7e,$60,$3c,$06,$66,$3c,$00 ; 5
	.db $00,$3c,$60,$7c,$66,$66,$3c,$00 ; 6
	.db $00,$3c,$7e,$06,$06,$0c,$0c,$00 ; 7
	.db $00,$3c,$66,$3c,$66,$66,$3c,$00 ; 8
	.db $00,$3c,$66,$66,$3e,$06,$3c,$00 ; 9
	.db $00,$00,$18,$18,$00,$18,$18,$00 ; colon
	.db $00,$00,$18,$18,$00,$18,$10,$00 ; semi-colon
	.db $00,$0e,$38,$60,$60,$38,$0e,$00 ; less than
	.db $00,$00,$00,$3c,$00,$3c,$00,$00 ; equals
	.db $00,$70,$1c,$06,$06,$1c,$70,$00 ; greater than
	.db $00,$3c,$66,$06,$1c,$00,$18,$00 ; question mark
	.db $00,$3c,$62,$5e,$5c,$60,$3c,$00 ; at
	.db $00,$3c,$66,$66,$7e,$66,$66,$00 ; A
	.db $00,$3c,$66,$7c,$66,$66,$3c,$00 ; B
	.db $00,$3c,$66,$60,$60,$66,$3c,$00 ; C
	.db $00,$38,$6c,$66,$66,$6c,$38,$00 ; D
	.db $00,$3c,$62,$78,$60,$62,$3c,$00 ; E
	.db $00,$3c,$66,$60,$78,$60,$20,$00 ; F
	.db $00,$3c,$66,$60,$6c,$66,$3c,$00 ; G
	.db $00,$24,$66,$7e,$66,$66,$24,$00 ; H
	.db $00,$3c,$3c,$18,$18,$3c,$3c,$00 ; I
	.db $00,$04,$06,$06,$66,$7e,$3c,$00 ; J
	.db $00,$26,$6e,$78,$6c,$66,$26,$00 ; K
	.db $00,$20,$60,$60,$60,$7e,$3c,$00 ; L
	.db $00,$42,$66,$7e,$7e,$66,$24,$00 ; M
	.db $00,$24,$76,$7e,$6e,$66,$24,$00 ; N
	.db $00,$3c,$66,$66,$66,$66,$3c,$00 ; O
	.db $00,$3c,$66,$66,$7c,$60,$20,$00 ; P
	.db $00,$3c,$66,$66,$66,$64,$3e,$00 ; Q
	.db $00,$3c,$66,$66,$7c,$6c,$26,$00 ; R
	.db $00,$3c,$60,$3c,$06,$66,$3c,$00 ; S
	.db $00,$3c,$7e,$18,$18,$18,$18,$00 ; T
	.db $00,$24,$66,$66,$66,$66,$3c,$00 ; U
	.db $00,$24,$66,$66,$66,$3c,$18,$00 ; V
	.db $00,$24,$66,$7e,$7e,$66,$42,$00 ; W
	.db $00,$66,$3c,$18,$3c,$66,$66,$00 ; X
	.db $00,$24,$66,$7e,$3c,$18,$18,$00 ; Y
	.db $00,$3e,$7c,$18,$30,$7e,$3c,$00 ; Z
	.db $00,$3c,$30,$30,$30,$30,$3c,$00 ; open square bracket
	.db $00,$60,$30,$18,$18,$0c,$06,$00 ; backslash
	.db $00,$3c,$0c,$0c,$0c,$0c,$3c,$00 ; close square bracket
	.db $00,$18,$3c,$66,$00,$00,$00,$00 ; tilde
	.db $00,$00,$00,$00,$00,$00,$7e,$00 ; underscore

	.db $00	; .... ....	; ........
	.db $7f	; .OOO OOO0	; .=======
	.db $7f	; .OOO OOO0	; .=======
	.db $00	; .... ....	; ........
	.db $00	; .... ....	; ........
	.db $7f	; .OOO OOO0	; .=======
	.db $7f	; .OOO OOO0	; .=======
	.db $00	; .... .... "="	; ........ ; chr 96
	
	.db $00	; .... ....	; ........
	.db $fe	; OOOO OOO.	; =======.
	.db $fe	; OOOO OOO.	; =======.
	.db $00	; .... ....	; ........
	.db $00	; .... ....	; ........
	.db $fe	; OOOO OOO.	; =======.
	.db $fe	; OOOO OOO.	; =======.
	.db $00	; .... .... "="	; ........ ; chr 97
charmask	
	.db $3c	; ..OO OO..
	.db $7e	; .OOO OOO.
	.db $ff	; OOOO OOOO
	.db $ff	; OOOO OOOO
	.db $ff	; OOOO OOOO
	.db $ff	; OOOO OOOO
	.db $7e	; .OOO OOO.
	.db $3c	; ..OO OO.. "O"	; ........ ; chr 98
;
; ==============================================================================
;	THE BASIC SECTION
;	========================================================================
Begin
	.db $00,$01		; Basic Line 1
	.dw Variables-RAND_USR	; Length of Basic Line 1
RAND_USR
	.db $F9,$D4,$26,$76	; RAND USR A N/L
;
;	======================================================= BASIC VARIABLES
dblINI  .equ (2*INIT)& $7FFF
Variables
	.db $66,$8F		;
	.db dblINI/256		;
	.db dblINI&255		;
	.db $00,$00		; A=INIT
;
;	=======================================================================
;
	.db $80
eof_Vars
	.end
